/*******************************************************************************

 * Copyright (c) 2011 itemis AG (http://www.itemis.eu) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package org.eclipse.xtext.xbase.compiler;

import java.util.List;

import org.eclipse.xtext.common.types.JvmType;

public class StringBuilderBasedAppendable implements IAppendable {

	private StringBuilder builder = new StringBuilder(8 * 1024);
	private int indentationlevel = 0;
	private String indentation = "  ";
	private ImportManager importManager;

	public IAppendable append(JvmType type) {
		appendType(type);
		return this;
	}

	public IAppendable append(String string) {
		String replaced = string.replace("\n", getIndentationString());
		builder.append(replaced);
		return this;
	}

	protected CharSequence getIndentationString() {
		StringBuilder sb = new StringBuilder(10);
		sb.append("\n");
		for (int i = 0; i < indentationlevel; i++) {
			sb.append(indentation);
		}
		return sb.toString();
	}

	@Override
	public String toString() {
		return builder.toString();
	}

	public IAppendable increaseIndentation() {
		indentationlevel++;
		return this;
	}

	public IAppendable decreaseIndentation() {
		if (indentationlevel == 0)
			throw new IllegalStateException("Can't reduce indentation level. It's already zero.");
		indentationlevel--;
		return this;
	}

	private ScopeStack scopes = new ScopeStack();
	
	public void setScopeStack(ScopeStack scopes) {
		this.scopes = scopes;
	}

	public StringBuilderBasedAppendable(ImportManager typeSerializer, String indentation) {
		this.importManager = typeSerializer;
		this.indentation = indentation;
		openScope();
	}

	public StringBuilderBasedAppendable(ImportManager typeSerializer) {
		this.importManager = typeSerializer;
		openScope();
	}

	public StringBuilderBasedAppendable() {
		this(new ImportManager(false));
	}

	public void openScope() {
		scopes.openScope(false);
	}
	
	public void openPseudoScope() {
		scopes.openScope(true);
	}
	
	public String declareVariable(Object key, String proposedName) {
		return scopes.declareVariable(key, proposedName, false);
	}
	
	public String declareSyntheticVariable(Object key, String proposedName) {
		return scopes.declareVariable(key, proposedName, true);
	}
	
	public void closeScope() {
		scopes.closeScope();
	}

	protected void appendType(final JvmType type) {
		importManager.appendType(type, builder);
	}

	public List<String> getImports() {
		return importManager.getImports();
	}

	protected ImportManager getImportManager() {
		return importManager;
	}

	public String getName(Object key) {
		return scopes.getName(key);
	}

	public Object getObject(String name) {
		return scopes.get(name);
	}
	
}