/*******************************************************************************
 * Copyright (c) 2005 - 2006 IBM Corporation & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    {Chris Aniszczyk} - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.wizards;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.codegen.jet.JETNature;
import org.eclipse.emf.codegen.jet.editor.JETEditorPlugin;
import org.eclipse.emf.codegen.jet.editor.JETUIMessages;
import org.eclipse.emf.codegen.jet.editor.presentation.JETEditor;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jdt.internal.corext.util.Messages;
import org.eclipse.jdt.internal.ui.dialogs.StatusInfo;
import org.eclipse.jdt.internal.ui.util.SWTUtil;
import org.eclipse.jdt.internal.ui.wizards.NewWizardMessages;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.views.navigator.ResourceSorter;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.part.FileEditorInput;

/**
 * @author Chris Aniszczyk
 * @since 1.0.0
 */
public class NewJETFileCreationWizardPage extends WizardPage {
	
	private IContainer container;
	private Label label;
	private GridData gridData;
	protected Text nameText;
	protected Text classText;
	protected Text locationText;
	protected Text packageText;
	protected Button locationButton;
	private IStatus rootStatus;
	
	protected NewJETFileCreationWizardPage(IContainer container) {
		super("NewJETFileCreationWizardPage");
		this.container = container;
		this.rootStatus = new StatusInfo(IStatus.ERROR, null);
		setTitle(JETUIMessages.NewJETFileCreationWizardPage_title);
		setDescription(JETUIMessages.NewJETFileCreationWizardPage_description);
	}
	
	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.verticalSpacing = 9;
		layout.makeColumnsEqualWidth = false;
		container.setLayout(layout);
		
		createNameComposite(container);
		createPackageComposite(container);
		createClassComposite(container);
		createLocationComposite(container);
		
		setControl(container);
		setMessage(null);
		validatePage();
		Dialog.applyDialogFont(container);
	}
	
	protected void createNameComposite(Composite container) {
		label = new Label(container, SWT.NONE);
		label.setText(JETUIMessages.NewJETFileCreationWizardPage_name_label_text);
		nameText = new Text(container, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		nameText.setLayoutData(gridData);
		nameText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validateName();
				validatePage();
			}
		});
	}
	
	protected void createPackageComposite(Composite container) {
		label = new Label(container, SWT.NONE);
		label.setText(JETUIMessages.NewJETFileCreationWizardPage_package_label_text);
		packageText = new Text(container, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		packageText.setLayoutData(gridData);
		packageText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validatePackage();
				validatePage();
			}
		});
	}
	
	protected void createClassComposite(Composite container) {
		label = new Label(container, SWT.NONE);
		label.setText(JETUIMessages.NewJETFileCreationWizardPage_ClassLabel);
		classText = new Text(container, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		classText.setLayoutData(gridData);
		classText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validateClass();
				validatePage();
			}
		});
	}
	
	protected void createLocationComposite(Composite container) {
		label = new Label(container, SWT.NONE);
		label.setText(JETUIMessages.NewJETFileCreationWizardPage_location_label_text);
		locationText = new Text(container, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.widthHint = 150;
		gridData.grabExcessHorizontalSpace = true;
		locationText.setEditable(false);
		locationText.setLayoutData(gridData);
		locationText.addModifyListener(new ModifyListener(){
			public void modifyText(ModifyEvent e){
				validateLocation();
				validatePage();
			}
		});
		locationButton = new Button(container, SWT.PUSH);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_END);
		gridData.widthHint = 50;
		locationButton.setLayoutData(gridData);
		locationButton.setText(JETUIMessages.NewJETFileCreationWizardPage_location_button_text); 
		locationButton.setToolTipText(JETUIMessages.NewJETFileCreationWizardPage_location_button_toolTip); 
		locationButton.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e) {
				handleFileLocation();
			}
		});
		SWTUtil.setButtonDimensionHint(locationButton);
	}
	
	protected void handleFileLocation() {
		final ElementTreeSelectionDialog dialog =
			new ElementTreeSelectionDialog(
					getShell(),
					new WorkbenchLabelProvider(),
					new WorkbenchContentProvider());
		dialog.setTitle(JETUIMessages.NewJETFileCreationWizardPage_location_dialog_title); 
		dialog.setMessage(JETUIMessages.NewJETFileCreationWizardPage_location_dialog_description); 
		dialog.setStatusLineAboveButtons(true);
		dialog.setEmptyListMessage(JETUIMessages.NewJETFileCreationWizardPage_emptyListMessage);
		dialog.setDoubleClickSelects(false);
		dialog.setAllowMultiple(false);
		dialog.addFilter(new ViewerFilter(){
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				if(element instanceof IProject) {
					JETNature nature = JETNature.getRuntime((IProject) element);
					if(nature != null)
						return true;
				}
				if(element instanceof IFolder) {
					IFolder folder = (IFolder) element;
					JETNature nature = JETNature.getRuntime(folder.getProject());
					
					List templateContainers = nature.getTemplateContainers();
					
					IContainer container = folder;
					while(container != null) {
						if(templateContainers.contains(container))
							return true;
						container = container.getParent();
					}
				}
				return false;
			}
		});
		
		dialog.setInput(ResourcesPlugin.getWorkspace().getRoot());
		dialog.setSorter(new ResourceSorter(ResourceSorter.NAME));
		
		dialog.setValidator(new ISelectionStatusValidator(){
			
			public IStatus validate(Object[] selection) {
				StatusInfo statusInfo = new StatusInfo();
				if(selection.length > 0 && selection[0] instanceof IFolder) {
					statusInfo.setOK();
					return statusInfo;
				}
				statusInfo.setError(JETUIMessages.NewJETFileCreationWizardPage_invalidTemplateContainer);
				return statusInfo;
			}		
		});
		
		dialog.setInitialSelection(container);
		if(dialog.open() == Window.OK) {
			Object[] elements = dialog.getResult();
			if (elements.length > 0){
				IResource elem = (IResource) elements[0];
				String newPath = getWorkspaceRelativePath(elem.getLocation().toString());
				locationText.setText(newPath + "/"); //$NON-NLS-1$
			}
		}
		
	}
	
	private String getWorkspaceRelativePath(String path){
		String workspacePath = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
		if(path.startsWith(workspacePath))
			path = path.replaceFirst(workspacePath, ""); //$NON-NLS-1$
		return path;
	}
	
	private void validatePage() { 
		setPageComplete(
				rootStatus.isOK() &&
				nameText.getText().length() > 0 &&
				classText.getText().length() > 0 && 
				packageText.getText().length() > 0 && 
				locationText.getText().length() > 0);
	}
	
	private void validateLocation() {
		StatusInfo status = new StatusInfo();
		if(locationText.getText().length() == 0) {
			status.setError(JETUIMessages.NewJETFileCreationWizardPage_invalidFileName);
		}
		updateStatus(status);
	}
	
	private void validatePackage() {
		StatusInfo status = new StatusInfo();
		String packName = packageText.getText();
		if (packName.length() > 0) {
			IStatus val = JavaConventions.validatePackageName(packName);
			if (val.getSeverity() == IStatus.ERROR) {
				status.setError(Messages.format(NewWizardMessages.NewPackageWizardPage_error_InvalidPackageName, val.getMessage())); 				
			} else if (val.getSeverity() == IStatus.WARNING) {
				status.setWarning(Messages.format(NewWizardMessages.NewPackageWizardPage_warning_DiscouragedPackageName, val.getMessage())); 
			}
		} else {
			status.setError(NewWizardMessages.NewPackageWizardPage_error_EnterName); 
		}
		updateStatus(status);
	}
	
	private void validateName() {
		StatusInfo status = new StatusInfo();
		if(nameText.getText().length() == 0) {
			status.setError(JETUIMessages.NewJETFileCreationWizardPage_invalidFileName);
		}
		updateStatus(status);
	}
	
	private void validateClass() {
		StatusInfo status = new StatusInfo();
		String typeName = classText.getText();
		// must not be empty
		if (typeName.length() == 0) {
			status.setError(NewWizardMessages.NewTypeWizardPage_error_EnterTypeName); 
		}	
		if (typeName.indexOf('.') != -1) {
			status.setError(NewWizardMessages.NewTypeWizardPage_error_QualifiedName); 
		}
		IStatus val = JavaConventions.validateJavaTypeName(typeName);
		if (val.getSeverity() == IStatus.ERROR) {
			status.setError(Messages.format(NewWizardMessages.NewTypeWizardPage_error_InvalidTypeName, val.getMessage())); 
		} else if (val.getSeverity() == IStatus.WARNING) {
			status.setWarning(Messages.format(NewWizardMessages.NewTypeWizardPage_warning_TypeNameDiscouraged, val.getMessage())); 
		}
		updateStatus(status);
	}
	
	private InputStream createJETStream() {
		StringWriter stringWriter = new StringWriter();
		
		String packageString = "package=\"" + packageText.getText() + "\" ";
		String classString = "class=\"" + classText.getText() + "\" ";
		
		stringWriter.write("<%@ jet " + packageString + classString + "%>");
		try {
			return new ByteArrayInputStream(stringWriter.toString().getBytes("UTF8")); } 
		catch (UnsupportedEncodingException e) {
			JETEditorPlugin.getDefault().log(e);
			return new ByteArrayInputStream(new byte[0]);
		}
	}
	
	public boolean finish() {
		IRunnableWithProgress operation = getOperation();
		try {
			getContainer().run(false, true, operation);
		} catch (InvocationTargetException e) {
			return false;
		} catch (InterruptedException e) {
			return false;
		}
		return true;
	}
	
	private IRunnableWithProgress getOperation() {
		IRunnableWithProgress operation = new WorkspaceModifyOperation() {
			public void execute(IProgressMonitor monitor) {
				String filePath = locationText.getText().concat(nameText.getText());
				IFile file = 
					ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(filePath));
				try {
					file.create(createJETStream(), true, monitor);
				} catch (CoreException e) {
					JETEditorPlugin.getDefault().log(e);
				}
				try {
					IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
					page.openEditor(new FileEditorInput(file), JETEditor.ID);
				} catch (PartInitException e) {
					JETEditorPlugin.getDefault().log(e);
				}
			}
		};
		return operation;
	}
	
	private void updateStatus(IStatus status) {
		int type = IMessageProvider.NONE;
		if(status.matches(IStatus.WARNING))
			type = IMessageProvider.WARNING;
		if(status.matches(IStatus.ERROR))
			type = IMessageProvider.ERROR;
		setMessage(status.getMessage(), type);
		this.rootStatus = status;
	}
	
}
