/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler

import lsat_graph.ActionTask
import lsat_graph.DispatchGroupTask
import org.eclipse.lsat.common.graph.directed.DirectedGraph
import org.eclipse.lsat.common.graph.directed.Edge
import org.eclipse.lsat.motioncalculator.MotionException
import org.eclipse.lsat.timing.util.ITimingCalculator
import org.eclipse.lsat.timing.util.SpecificationException
import org.eclipse.lsat.common.scheduler.graph.Task
import org.eclipse.lsat.common.scheduler.graph.TaskDependencyGraph
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor
import org.slf4j.LoggerFactory
import java.math.BigDecimal
import org.eclipse.emf.common.util.BasicEList
import lsat_graph.PeripheralActionTask

@FinalFieldsConstructor
class AddExecutionData {
    static val LOGGER = LoggerFactory.getLogger(AddExecutionData)

    extension val ITimingCalculator timingCalculator;

    def <T extends Task> TaskDependencyGraph<T> transformModel(
        TaskDependencyGraph<T> graph, String[] filter) throws SpecificationException, MotionException {
        LOGGER.debug('Starting transformation')
        graph.addExecutionData(filter)
        LOGGER.debug('Finished transformation')
        return graph
    }

    private def <T extends Task, E extends Edge> void addExecutionData(
        DirectedGraph<T, E> graph, String[] filter) throws SpecificationException, MotionException {
        graph.subGraphs.forEach[addExecutionData(filter)]
        graph.nodes.filter(PeripheralActionTask)
            .filter[filter.length == 0 || filter.contains(action.fqn)]
            .forEach[doAddExecutionData]
    }

    private def dispatch void doAddExecutionData(DispatchGroupTask task) {
        if (task.executionData === null) {
            throw new IllegalStateException('''Expected execution data to be set for task: «task»''')
        }
    }

    private def dispatch void doAddExecutionData(ActionTask<?> task) throws SpecificationException, MotionException {
        for (entry : task.action.calculateMotionData.entrySet) {
            for (timeDataValues: entry.value.timeData) {
                val timeDataBigDecimal = new BasicEList<BigDecimal>()
                for (d : timeDataValues) {
                    timeDataBigDecimal.add(BigDecimal::valueOf(d))
                }
                task.addExecutionData(entry.key, timeDataBigDecimal)
            }

            for (paramName: entry.value.parameterNames) {
                if (!task.executionDataParameters.contains(paramName)) {
                    task.executionDataParameters.add(paramName)
                }
            }
        }
    }
}
