//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2023, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.plcgen.targets;

import java.util.EnumSet;
import java.util.List;

import org.eclipse.escet.cif.metamodel.cif.declarations.Constant;
import org.eclipse.escet.cif.plcgen.conversion.ModelTextGenerator;
import org.eclipse.escet.cif.plcgen.generators.CifProcessor;
import org.eclipse.escet.cif.plcgen.generators.ContinuousVariablesGenerator;
import org.eclipse.escet.cif.plcgen.generators.PlcCodeStorage;
import org.eclipse.escet.cif.plcgen.generators.PlcVariablePurpose;
import org.eclipse.escet.cif.plcgen.generators.TransitionGenerator;
import org.eclipse.escet.cif.plcgen.generators.VariableStorage;
import org.eclipse.escet.cif.plcgen.generators.io.DefaultIoAddress;
import org.eclipse.escet.cif.plcgen.generators.io.IoAddress;
import org.eclipse.escet.cif.plcgen.generators.io.IoDirection;
import org.eclipse.escet.cif.plcgen.model.declarations.PlcPou;
import org.eclipse.escet.cif.plcgen.model.declarations.PlcProject;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcIntLiteral;
import org.eclipse.escet.cif.plcgen.model.expressions.PlcRealLiteral;
import org.eclipse.escet.cif.plcgen.model.functions.PlcBasicFuncDescription.ExprAssociativity;
import org.eclipse.escet.cif.plcgen.model.functions.PlcBasicFuncDescription.ExprBinding;
import org.eclipse.escet.cif.plcgen.model.functions.PlcBasicFuncDescription.PlcFuncNotation;
import org.eclipse.escet.cif.plcgen.model.functions.PlcBasicFuncDescription.PlcFuncTypeExtension;
import org.eclipse.escet.cif.plcgen.model.functions.PlcFuncOperation;
import org.eclipse.escet.cif.plcgen.model.statements.PlcFuncApplStatement;
import org.eclipse.escet.cif.plcgen.model.types.PlcElementaryType;
import org.eclipse.escet.cif.plcgen.model.types.PlcType;
import org.eclipse.escet.cif.plcgen.options.ConvertEnums;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.exceptions.InputOutputException;
import org.eclipse.escet.common.java.exceptions.InvalidInputException;

/** Code generator 'interface' for a {@link PlcBaseTarget}. */
public abstract class PlcTarget {
    /**
     * Obtain the target type of the target.
     *
     * @return The target type of the target.
     */
    public abstract PlcTargetType getTargetType();

    /**
     * Retrieve the converter of the PLC model classes to their textual equivalent.
     *
     * @return The text generator for the PLC model classes.
     */
    public abstract ModelTextGenerator getModelTextGenerator();

    /**
     * Retrieve the CIF processor.
     *
     * @return The CIF processor of the target.
     */
    public abstract CifProcessor getCifProcessor();

    /**
     * Retrieve the transition generator.
     *
     * @return The transition generator of the target.
     */
    public abstract TransitionGenerator getTransitionGenerator();

    /**
     * Retrieve the continuous variables generator.
     *
     * @return The continuous variables generator of the target.
     */
    public abstract ContinuousVariablesGenerator getContinuousVariablesGenerator();

    /**
     * Retrieve the variable storage.
     *
     * @return The variable storage of the target.
     */
    public abstract VariableStorage getVarStorage();

    /**
     * Retrieve the PLC code storage.
     *
     * @return The PLC code storage of the target.
     */
    public abstract PlcCodeStorage getCodeStorage();

    /**
     * Get whether the given name is allowed by the PLC language and the target.
     *
     * @param name The name to check.
     * @return {@code true} if the name is allowed, {@code false} otherwise.
     */
    public abstract boolean isAllowedName(String name);

    /**
     * Get the text to use in the code for accessing a variable with the given name and purpose.
     *
     * @param purpose Purpose of the variable.
     * @param varName Name of the variable.
     * @return The text to use for reading or writing the variable in the PLC code.
     */
    public abstract String getUsageVariableText(PlcVariablePurpose purpose, String varName);

    /**
     * Get the name to use to call the {@code TON} function within the instance variable of the block function.
     *
     * @return The name to use to call the {@code TON} function within the instance variable of the block function.
     */
    public abstract String getTonFuncBlockCallName();

    /**
     * Define where to store the persistent state variables.
     *
     * @return The storage location of persistent state variables.
     */
    public abstract StateVariableStorage getStateVariableStorage();

    /** Storage location of persistent state variables. */
    public enum StateVariableStorage {
        /** Store state variables in a global variable table in a resource. */
        STATE_VARS_IN_GLOBAL_TABLE,

        /**
         * Store state variables in {@link PlcPou#persistedVars}, the persistent local variables of the main program.
         */
        STATE_VARS_IN_MAIN,
    }

    /**
     * Returns whether the target supports arrays.
     *
     * @return Whether arrays are supported.
     */
    public abstract boolean supportsArrays();

    /**
     * Returns whether or not the PLC target type supports the given constant.
     *
     * @param constant The constant to consider.
     * @return Whether the constant is supported.
     */
    public abstract boolean supportsConstant(Constant constant);

    /**
     * Return how to convert enumerations.
     *
     * @return The desired conversion to enumerations. Never returns {@link ConvertEnums#AUTO}.
     */
    public abstract ConvertEnums getActualEnumerationsConversion();

    /**
     * Does the target support the given semantic operation?
     *
     * @param funcOper Semantic operation being queried.
     * @param numArgs Number of supplied arguments to the applied function.
     * @return Whether the target supports the given operation.
     * @see #getSupportedFuncNotations
     */
    public final boolean supportsOperation(PlcFuncOperation funcOper, int numArgs) {
        return !getSupportedFuncNotations(funcOper, numArgs).isEmpty();
    }

    /**
     * Get the set of supported function-call notations for the given semantic operation.
     *
     * <p>
     * Notes:
     * <ul>
     * <li>Remove operations by not having a notation for them. This only works if the code generator can fallback to
     * another way to express the needed functionality. Currently that is only implemented for LOG.</li>
     * <li>The {@link PlcFuncApplStatement} class needs a function in prefix notation.</li>
     * </ul>
     * </p>
     *
     * @param funcOper Semantic operation being queried.
     * @param numArgs Number of supplied arguments to the applied function.
     * @return The set of supported function-call notations for the operation.
     * @see #supportsOperation
     */
    public abstract EnumSet<PlcFuncNotation> getSupportedFuncNotations(PlcFuncOperation funcOper, int numArgs);

    /**
     * Returns the priority level of the given expression binding. Smaller values bind stronger.
     *
     * @param exprBinding The expression binding.
     * @return The priority level.
     */
    public abstract int getExprPriority(ExprBinding exprBinding);

    /**
     * Returns the associativity of the given expression binding.
     *
     * @param exprBinding The expression binding.
     * @return The associativity.
     */
    public abstract ExprAssociativity getExprAssociativity(ExprBinding exprBinding);

    /**
     * Get the rule for extending a function with a type, for example {@code SEL} to {@code SEL_INT}.
     *
     * @param funcOper Semantic operation being queried.
     * @return The rule for extending the function with a type.
     */
    public abstract PlcFuncTypeExtension getTypeExtension(PlcFuncOperation funcOper);

    /**
     * Retrieve the supported integer types of the target, ordered by increasing size.
     *
     * @return The supported integer types of the target, ordered by increasing size.
     */
    public abstract List<PlcElementaryType> getSupportedIntegerTypes();

    /**
     * Get the size of the largest supported integer type.
     *
     * @return Number of bits used for storing the largest supported integer type.
     */
    public abstract int getMaxIntegerTypeSize();

    /**
     * Get the type of a standard integer value in the PLC.
     *
     * <p>
     * As CIF uses signed 32 bit integer, a {@code DINT} is recommended.
     * </p>
     *
     * @return The type of a standard integer value in the PLC.
     */
    public abstract PlcElementaryType getStdIntegerType();

    /**
     * Construct a new standard integer literal with the given value.
     *
     * @param value Value of the new standard integer literal.
     * @return The created literal.
     */
    public PlcIntLiteral makeStdInteger(int value) {
        return new PlcIntLiteral(value, getStdIntegerType());
    }

    /**
     * Retrieve the supported real types of the target, ordered by increasing size.
     *
     * @return The supported real types of the target, ordered by increasing size.
     */
    public abstract List<PlcElementaryType> getSupportedRealTypes();

    /**
     * Get the size of the largest supported real type.
     *
     * @return Number of bits used for storing the largest supported real type.
     */
    public abstract int getMaxRealTypeSize();

    /**
     * Get the type of a standard real value in the PLC.
     *
     * <p>
     * As CIF uses 64 bit reals, an {@code LREAL} is recommended.
     * </p>
     *
     * @return The type of a standard real value in the PLC.
     */
    public abstract PlcElementaryType getStdRealType();

    /**
     * Construct a new standard real literal with the given value.
     *
     * @param value Value of the new standard real literal.
     * @return The created literal.
     */
    public PlcRealLiteral makeStdReal(String value) {
        return makeRealValue(value, getStdRealType());
    }

    /**
     * Construct a real value literal for the given real PLC type.
     *
     * <p>
     * The returned value may deviate from the given value due to rounding in the conversion to the given real type.
     * </p>
     *
     * @param value Real value to convert.
     * @param type PLC type to use for the value.
     * @return The real literal value that is equal or close to the given value, for the given PLC type.
     */
    public PlcRealLiteral makeRealValue(String value, PlcType type) {
        if (type == PlcElementaryType.LREAL_TYPE) {
            // Convert value text to double and back to text to make the value expressed in the text fit in the LREAL
            // type (which is the same format as a double). CIF doubles don't allow infinity, thus that result text
            // should never happen.
            Double realValue = Double.parseDouble(value);
            Assert.check(!realValue.isNaN());
            Assert.check(realValue != Double.POSITIVE_INFINITY && realValue != Double.NEGATIVE_INFINITY);
            realValue = (realValue == -0.0) ? 0.0 : realValue;
            return new PlcRealLiteral(realValue.toString(), type);
        } else if (type == PlcElementaryType.REAL_TYPE) {
            // Convert value text to float and back to text to make the value expressed in the text fit in the REAL
            // type (which is the same format as a float). As a float has smaller upper and lower bounds than a CIF
            // real, infinity may happen and these need to be converted to min/max float values. Similarly, very small
            // negative values may need to be converted to 0.0.
            Float realValue = Float.parseFloat(value);
            Assert.check(!realValue.isNaN());
            realValue = (realValue == Float.POSITIVE_INFINITY) ? Float.MAX_VALUE : realValue;
            realValue = (realValue == Float.NEGATIVE_INFINITY) ? Float.MIN_VALUE : realValue;
            realValue = (realValue == -0.0f) ? 0.0f : realValue;
            return new PlcRealLiteral(realValue.toString(), type);
        } else {
            throw new AssertionError("Unexpected type \"" + type.toString() + "\" in real literal conversion.");
        }
    }

    /**
     * Retrieve the supported bit string types of the target, ordered by increasing size.
     *
     * @return The supported bit string types of the target, ordered by increasing size.
     */
    public abstract List<PlcElementaryType> getSupportedBitStringTypes();

    /**
     * Parse a PLC I/O address.
     *
     * @param plcAddressText Text to parse.
     * @return The parsed address information and its properties or {@code null} if the text cannot be parsed.
     */
    public IoAddress parseIoAddress(String plcAddressText) {
        return DefaultIoAddress.parseAddress(plcAddressText);
    }

    /**
     * Verify that the given I/O table entry is acceptable to the target.
     *
     * <p>
     * If the entry is not acceptable, it should be reported to the user with an {@link InvalidInputException}.
     * </p>
     *
     * @param address The I/O address to verify.
     * @param plcTableType Type of the I/O data being transferred.
     * @param directionFromCif Direction of the I/O table entry.
     * @param ioName User-provided name of the I/O variable, may be {@code null}.
     * @param tableLinePositionText Text describing the table line for this entry, to use for reporting an error. The
     *     text is {@code "at line ... of I/O table file \"...\""}.
     * @throws InputOutputException If the provided entry is not acceptable to the target.
     */
    public abstract void verifyIoTableEntry(IoAddress address, PlcType plcTableType, IoDirection directionFromCif,
            String ioName, String tableLinePositionText);

    /**
     * Check whether a user-supplied I/O variable name is acceptable to the target.
     *
     * @param name Name of the I/O variable to check.
     * @return Whether the given name is acceptable to the target as a name for an I/O variable.
     */
    public abstract boolean checkIoVariableName(String name);

    /**
     * Get replacement string for the CIF input file extension including dot, used to derive an output path.
     *
     * @return The replacement string.
     */
    public abstract String getPathSuffixReplacement();

    /**
     * Write the project to the output.
     *
     * @param project Project to write.
     * @note Depending on the actual write implementation a single file or a directory may be written.
     */
    public abstract void writeOutput(PlcProject project);
}
