/**
 ********************************************************************************
 * Copyright (c) 2015-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters072.impl;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.converters.common.ServiceConstants;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters.common.base.IConverter;
import org.eclipse.app4mc.amalthea.converters.common.converter.AbstractConverter;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.jdom2.Document;
import org.jdom2.Element;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is responsible for converting the root element of the model file as : "Amalthea" (only if for the supplied
 * model root element is other than "Amalthea" e.g: hwModel or swModel)
 *
 * @author mez2rng
 *
 */
@Component(
		property = {
			ServiceConstants.INPUT_MODEL_VERSION_PROPERTY + "=0.7.1",
			ServiceConstants.OUTPUT_MODEL_VERSION_PROPERTY + "=0.7.2",
			"service.ranking:Integer=100"},
		service = IConverter.class)

public class RootElementConverter extends AbstractConverter {

	private static final Logger LOGGER = LoggerFactory.getLogger(RootElementConverter.class);

	@Override
	@Activate
	protected void activate(Map<String, Object> properties) {
		super.activate(properties);
	}

	@Override
	public void convert(File targetFile, Map<File, Document> filename2documentMap, List<ICache> caches) {

		/*-
		 * ==============
		 *|| Input Model :||
		 * ==============
		 * <am:MappingModel xmi:version="2.0" xmlns:xmi="http://www.omg.org/XMI" xmlns:am="http://app4mc.eclipse.org/amalthea/0.7.1" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
		 *	</am:MappingModel>
		 *
		 * ==============
		 *|| Output Model :||
		 * ==============
		 *
		 * <am:Amalthea xmlns:am="http://app4mc.eclipse.org/amalthea/0.7.2" xmlns:xmi="http://www.omg.org/XMI" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
		 *		<mappingModel>
		 *
		 *		</mappingModel>
		  </am:Amalthea>
		 */


		LOGGER.info("Migration from 0.7.1 to 0.7.2 : Executing Root Element Converter for model file : {}",
				targetFile.getName());

		final Document document = filename2documentMap.get(targetFile);

		if (document == null) {
			return;
		}
		final Element rootElement = document.getRootElement();

		final String rootTagName = rootElement.getName();

		if (!rootTagName.equals("Amalthea")) {

			if (rootTagName.equals("SWModel") || rootTagName.equals("HWModel") || rootTagName.equals("OSModel")
					|| rootTagName.equals("StimuliModel") || rootTagName.equals("ConstraintsModel")
					|| rootTagName.equals("EventModel") || rootTagName.equals("PropertyConstraintsModel")
					|| rootTagName.equals("MappingModel") || rootTagName.equals("ConfigModel")
					|| rootTagName.equals("ComponentsModel")) {

				final Element newRootElement = new Element("Amalthea");

				HelperUtil.copyAllNameSpaces(rootElement, newRootElement);

				updateCurrentRootTagProps(rootElement);

				document.removeContent();

				newRootElement.addContent(rootElement);

				document.addContent(newRootElement);
			}
		}
	}

	/**
	 * This method is used to remove the default namespaces associated to the Root Element and also change the name of
	 * it -> so that it can be added as a sub-tag inside "Amalthea root element"
	 *
	 * @param rootElement
	 *            Element This is the root element of the model file (where Amalthea is not root element)
	 */
	private void updateCurrentRootTagProps(final Element rootElement) {

		HelperUtil.removeDefaultAttribs(rootElement);

		final String name = rootElement.getName();

		if (name.equals("SWModel")) {
			rootElement.setName("swModel");
		}
		else if (name.equals("HWModel")) {
			rootElement.setName("hwModel");
		}
		else if (name.equals("OSModel")) {
			rootElement.setName("osModel");
		}
		else if (name.equals("StimuliModel")) {
			rootElement.setName("stimuliModel");
		}
		else if (name.equals("ConstraintsModel")) {
			rootElement.setName("constraintsModel");
		}
		else if (name.equals("EventModel")) {
			rootElement.setName("eventModel");
		}
		else if (name.equals("PropertyConstraintsModel")) {
			rootElement.setName("propertyConstraintsModel");
		}
		else if (name.equals("MappingModel")) {
			rootElement.setName("mappingModel");
		}
		else if (name.equals("ConfigModel")) {
			rootElement.setName("configModel");
		}
		else if (name.equals("ComponentsModel")) {
			rootElement.setName("componentsModel");
		}
	}

}
