﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/ConsistencyLevel.h>
#include <aws/clouddirectory/model/ObjectReference.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class ListAttachedIndicesRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API ListAttachedIndicesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAttachedIndices"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the directory.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  ListAttachedIndicesRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A reference to the object that has indices attached.</p>
   */
  inline const ObjectReference& GetTargetReference() const { return m_targetReference; }
  inline bool TargetReferenceHasBeenSet() const { return m_targetReferenceHasBeenSet; }
  template <typename TargetReferenceT = ObjectReference>
  void SetTargetReference(TargetReferenceT&& value) {
    m_targetReferenceHasBeenSet = true;
    m_targetReference = std::forward<TargetReferenceT>(value);
  }
  template <typename TargetReferenceT = ObjectReference>
  ListAttachedIndicesRequest& WithTargetReference(TargetReferenceT&& value) {
    SetTargetReference(std::forward<TargetReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAttachedIndicesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to retrieve.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAttachedIndicesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The consistency level to use for this operation.</p>
   */
  inline ConsistencyLevel GetConsistencyLevel() const { return m_consistencyLevel; }
  inline bool ConsistencyLevelHasBeenSet() const { return m_consistencyLevelHasBeenSet; }
  inline void SetConsistencyLevel(ConsistencyLevel value) {
    m_consistencyLevelHasBeenSet = true;
    m_consistencyLevel = value;
  }
  inline ListAttachedIndicesRequest& WithConsistencyLevel(ConsistencyLevel value) {
    SetConsistencyLevel(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;
  bool m_directoryArnHasBeenSet = false;

  ObjectReference m_targetReference;
  bool m_targetReferenceHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  ConsistencyLevel m_consistencyLevel{ConsistencyLevel::NOT_SET};
  bool m_consistencyLevelHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
