﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glacier/GlacierRequest.h>
#include <aws/glacier/Glacier_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Glacier {
namespace Model {

/**
 * <p>Provides options to upload a part of an archive in a multipart upload
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glacier-2012-06-01/UploadMultipartPartInput">AWS
 * API Reference</a></p>
 */
class UploadMultipartPartRequest : public StreamingGlacierRequest {
 public:
  AWS_GLACIER_API UploadMultipartPartRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UploadMultipartPart"; }

  AWS_GLACIER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>AccountId</code> value is the AWS account ID of the account that
   * owns the vault. You can either specify an AWS account ID or optionally a single
   * '<code>-</code>' (hyphen), in which case Amazon Glacier uses the AWS account ID
   * associated with the credentials used to sign the request. If you use an account
   * ID, do not include any hyphens ('-') in the ID. </p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  UploadMultipartPartRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the vault.</p>
   */
  inline const Aws::String& GetVaultName() const { return m_vaultName; }
  inline bool VaultNameHasBeenSet() const { return m_vaultNameHasBeenSet; }
  template <typename VaultNameT = Aws::String>
  void SetVaultName(VaultNameT&& value) {
    m_vaultNameHasBeenSet = true;
    m_vaultName = std::forward<VaultNameT>(value);
  }
  template <typename VaultNameT = Aws::String>
  UploadMultipartPartRequest& WithVaultName(VaultNameT&& value) {
    SetVaultName(std::forward<VaultNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The upload ID of the multipart upload.</p>
   */
  inline const Aws::String& GetUploadId() const { return m_uploadId; }
  inline bool UploadIdHasBeenSet() const { return m_uploadIdHasBeenSet; }
  template <typename UploadIdT = Aws::String>
  void SetUploadId(UploadIdT&& value) {
    m_uploadIdHasBeenSet = true;
    m_uploadId = std::forward<UploadIdT>(value);
  }
  template <typename UploadIdT = Aws::String>
  UploadMultipartPartRequest& WithUploadId(UploadIdT&& value) {
    SetUploadId(std::forward<UploadIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SHA256 tree hash of the data being uploaded.</p>
   */
  inline const Aws::String& GetChecksum() const { return m_checksum; }
  inline bool ChecksumHasBeenSet() const { return m_checksumHasBeenSet; }
  template <typename ChecksumT = Aws::String>
  void SetChecksum(ChecksumT&& value) {
    m_checksumHasBeenSet = true;
    m_checksum = std::forward<ChecksumT>(value);
  }
  template <typename ChecksumT = Aws::String>
  UploadMultipartPartRequest& WithChecksum(ChecksumT&& value) {
    SetChecksum(std::forward<ChecksumT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Identifies the range of bytes in the assembled archive that will be uploaded
   * in this part. Amazon Glacier uses this information to assemble the archive in
   * the proper sequence. The format of this header follows RFC 2616. An example
   * header is Content-Range:bytes 0-4194303/ *.</p>
   */
  inline const Aws::String& GetRange() const { return m_range; }
  inline bool RangeHasBeenSet() const { return m_rangeHasBeenSet; }
  template <typename RangeT = Aws::String>
  void SetRange(RangeT&& value) {
    m_rangeHasBeenSet = true;
    m_range = std::forward<RangeT>(value);
  }
  template <typename RangeT = Aws::String>
  UploadMultipartPartRequest& WithRange(RangeT&& value) {
    SetRange(std::forward<RangeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_vaultName;

  Aws::String m_uploadId;

  Aws::String m_checksum;

  Aws::String m_range;

  bool m_accountIdHasBeenSet = false;
  bool m_vaultNameHasBeenSet = false;
  bool m_uploadIdHasBeenSet = false;
  bool m_checksumHasBeenSet = false;
  bool m_rangeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glacier
}  // namespace Aws
