# frozen_string_literal: true

module Sidebars
  module Groups
    module Menus
      class WorkItemsMenu < ::Sidebars::Menu
        include Gitlab::Utils::StrongMemoize
        include Sidebars::Concerns::ShowWorkItemsBadge

        override :configure_menu_items
        def configure_menu_items
          return unless can?(context.current_user, :read_group_issues, context.group)

          add_item(list_menu_item)
          add_item(boards_menu_item)
          add_item(milestones_menu_item)

          true
        end

        override :title
        def title
          s_('WorkItem|Work items')
        end

        override :sprite_icon
        def sprite_icon
          'issues'
        end

        override :has_pill?
        def has_pill?
          false
        end

        override :active_routes
        def active_routes
          {
            path: %w[
              groups/issues#index groups/issues#show groups/issues#new
              groups/work_items#index groups/work_items#show groups/work_items#new
            ]
          }
        end

        override :pill_count_field
        def pill_count_field
          'openIssuesCount'
        end

        override :pill_html_options
        def pill_html_options
          {
            class: 'issue_counter'
          }
        end

        override :serialize_as_menu_item_args
        def serialize_as_menu_item_args
          super.merge({
            active_routes: list_menu_item.active_routes,
            pill_count: pill_count,
            pill_count_field: pill_count_field,
            has_pill: has_pill?,
            super_sidebar_parent: ::Sidebars::Groups::SuperSidebarMenus::PlanMenu,
            item_id: :group_issue_list,
            badge: work_items_badge
          })
        end

        private

        def list_menu_item
          ::Sidebars::MenuItem.new(
            title: s_('WorkItem|Work items'),
            link: group_work_items_path(context.group),
            super_sidebar_parent: ::Sidebars::NilMenuItem,
            active_routes: active_routes,
            container_html_options: { aria: { label: s_('WorkItem|Work items') } },
            item_id: :issue_list
          )
        end

        def boards_menu_item
          unless can?(context.current_user, :read_group_boards, context.group)
            return ::Sidebars::NilMenuItem.new(item_id: :boards)
          end

          title = if context.is_super_sidebar
                    context.group.multiple_issue_boards_available? ? s_('Issue boards') : s_('Issue board')
                  else
                    context.group.multiple_issue_boards_available? ? s_('Boards|Boards') : s_('Boards|Board')
                  end

          ::Sidebars::MenuItem.new(
            title: title,
            link: group_boards_path(context.group),
            super_sidebar_parent: ::Sidebars::Groups::SuperSidebarMenus::PlanMenu,
            active_routes: { path: %w[boards#index boards#show] },
            item_id: context.is_super_sidebar ? :issue_boards : :boards
          )
        end

        def milestones_menu_item
          unless can?(context.current_user, :read_group_milestones, context.group)
            return ::Sidebars::NilMenuItem.new(item_id: :milestones)
          end

          ::Sidebars::MenuItem.new(
            title: _('Milestones'),
            link: group_milestones_path(context.group),
            super_sidebar_parent: ::Sidebars::Groups::SuperSidebarMenus::PlanMenu,
            active_routes: { path: 'milestones#index' },
            item_id: :milestones
          )
        end

        def work_items_badge
          return unless show_work_items_badge?

          {
            label: _('New'),
            tooltip: work_items_badge_tooltip_text
          }
        end

        def work_items_badge_tooltip_text
          if context.container.licensed_feature_available?(:epics)
            s_('WorkItem|Epics and issues are now work items.')
          else
            s_('WorkItem|Issues are now work items.')
          end
        end
      end
    end
  end
end

Sidebars::Groups::Menus::WorkItemsMenu.prepend_mod_with('Sidebars::Groups::Menus::WorkItemsMenu')
