//=================================================================================================
/*!
//  \file blaze/math/expressions/DVecDVecMapExpr.h
//  \brief Header file for the dense vector/dense vector map expression
//
//  Copyright (C) 2012-2020 Klaus Iglberger - All Rights Reserved
//
//  This file is part of the Blaze library. You can redistribute it and/or modify it under
//  the terms of the New (Revised) BSD License. Redistribution and use in source and binary
//  forms, with or without modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright notice, this list of
//     conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright notice, this list
//     of conditions and the following disclaimer in the documentation and/or other materials
//     provided with the distribution.
//  3. Neither the names of the Blaze development group nor the names of its contributors
//     may be used to endorse or promote products derived from this software without specific
//     prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
//  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
//  SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
//  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
//  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
//  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
*/
//=================================================================================================

#ifndef _BLAZE_MATH_EXPRESSIONS_DVECDVECMAPEXPR_H_
#define _BLAZE_MATH_EXPRESSIONS_DVECDVECMAPEXPR_H_


//*************************************************************************************************
// Includes
//*************************************************************************************************

#include <iterator>
#include <utility>
#include <blaze/math/Aliases.h>
#include <blaze/math/constraints/DenseVector.h>
#include <blaze/math/constraints/RequiresEvaluation.h>
#include <blaze/math/constraints/TransposeFlag.h>
#include <blaze/math/Exception.h>
#include <blaze/math/expressions/Computation.h>
#include <blaze/math/expressions/DenseVector.h>
#include <blaze/math/expressions/Forward.h>
#include <blaze/math/expressions/VecVecMapExpr.h>
#include <blaze/math/functors/And.h>
#include <blaze/math/functors/Atan2.h>
#include <blaze/math/functors/Bitand.h>
#include <blaze/math/functors/Bitor.h>
#include <blaze/math/functors/Bitxor.h>
#include <blaze/math/functors/Hypot.h>
#include <blaze/math/functors/Join.h>
#include <blaze/math/functors/MakePair.h>
#include <blaze/math/functors/Max.h>
#include <blaze/math/functors/Min.h>
#include <blaze/math/functors/Or.h>
#include <blaze/math/functors/Pow.h>
#include <blaze/math/functors/ShiftLV.h>
#include <blaze/math/functors/ShiftRV.h>
#include <blaze/math/shims/Serial.h>
#include <blaze/math/SIMD.h>
#include <blaze/math/traits/MapTrait.h>
#include <blaze/math/typetraits/HasLoad.h>
#include <blaze/math/typetraits/IsAligned.h>
#include <blaze/math/typetraits/IsExpression.h>
#include <blaze/math/typetraits/IsPadded.h>
#include <blaze/math/typetraits/IsPaddingEnabled.h>
#include <blaze/math/typetraits/IsSIMDEnabled.h>
#include <blaze/math/typetraits/IsVector.h>
#include <blaze/math/typetraits/RequiresEvaluation.h>
#include <blaze/system/HostDevice.h>
#include <blaze/system/Inline.h>
#include <blaze/system/MacroDisable.h>
#include <blaze/util/Assert.h>
#include <blaze/util/EnableIf.h>
#include <blaze/util/FunctionTrace.h>
#include <blaze/util/IntegralConstant.h>
#include <blaze/util/mpl/If.h>
#include <blaze/util/Types.h>


namespace blaze {

//=================================================================================================
//
//  CLASS DVECDVECMAPEXPR
//
//=================================================================================================

//*************************************************************************************************
/*!\brief Expression object for the dense vector-dense vector map() function.
// \ingroup dense_vector_expression
//
// The DVecDVecMapExpr class represents the compile time expression for the pairwise evaluation
// of a binary custom operation on the elements of two dense vectors via the map() function.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , typename OP   // Type of the custom operation
        , bool TF >     // Transpose flag
class DVecDVecMapExpr
   : public VecVecMapExpr< DenseVector< DVecDVecMapExpr<VT1,VT2,OP,TF>, TF > >
   , private Computation
{
 private:
   //**Type definitions****************************************************************************
   using RT1 = ResultType_t<VT1>;     //!< Result type of the left-hand side dense vector expression.
   using RT2 = ResultType_t<VT2>;     //!< Result type of the right-hand side dense vector expression.
   using ET1 = ElementType_t<VT1>;    //!< Element type of the left-hand side dense vector expression.
   using ET2 = ElementType_t<VT2>;    //!< Element type of the right-hand side dense vector expression.
   using RN1 = ReturnType_t<VT1>;     //!< Return type of the left-hand side dense vector expression.
   using RN2 = ReturnType_t<VT2>;     //!< Return type of the right-hand side dense vector expression.
   using CT1 = CompositeType_t<VT1>;  //!< Composite type of the left-hand side dense vector expression.
   using CT2 = CompositeType_t<VT2>;  //!< Composite type of the right-hand side dense vector expression.
   //**********************************************************************************************

   //**Serial evaluation strategy******************************************************************
   //! Compilation switch for the serial evaluation strategy of the map expression.
   /*! The \a useAssign compile time constant expression represents a compilation switch for the
       serial evaluation strategy of the map expression. In case either of the two dense vector
       operands requires an intermediate evaluation, \a useAssign will be set to \a true and the
       map expression will be evaluated via the \a assign function family. Otherwise \a useAssign
       will be set to \a false and the expression will be evaluated via the subscript operator. */
   static constexpr bool useAssign = ( RequiresEvaluation_v<VT1> || RequiresEvaluation_v<VT2> );

   /*! \cond BLAZE_INTERNAL */
   //! Helper variable template for the explicit application of the SFINAE principle.
   template< typename VT >
   static constexpr bool UseAssign_v = useAssign;
   /*! \endcond */
   //**********************************************************************************************

   //**Parallel evaluation strategy****************************************************************
   /*! \cond BLAZE_INTERNAL */
   //! Helper variable template for the explicit application of the SFINAE principle.
   /*! This variable template is a helper for the selection of the parallel evaluation strategy.
       In case at least one of the two dense vector operands is not SMP assignable and at least
       one of the two operands requires an intermediate evaluation, the variable is set to \a true
       and the expression specific evaluation strategy is selected. Otherwise the variable is set
       to \a false and the default strategy is chosen. */
   template< typename VT >
   static constexpr bool UseSMPAssign_v =
      ( ( !VT1::smpAssignable || !VT2::smpAssignable ) && useAssign );
   /*! \endcond */
   //**********************************************************************************************

 public:
   //**Type definitions****************************************************************************
   //! Type of this DVecDVecMapExpr instance.
   using This = DVecDVecMapExpr<VT1,VT2,OP,TF>;

   //! Base type of this DVecDVecMapExpr instance.
   using BaseType = VecVecMapExpr< DenseVector<This,TF> >;

   using ResultType    = MapTrait_t<RT1,RT2,OP>;       //!< Result type for expression template evaluations.
   using TransposeType = TransposeType_t<ResultType>;  //!< Transpose type for expression template evaluations.
   using ElementType   = ElementType_t<ResultType>;    //!< Resulting element type.

   //! Return type for expression template evaluations.
   using ReturnType = decltype( std::declval<OP>()( std::declval<RN1>(), std::declval<RN2>() ) );

   //! Data type for composite expression templates.
   using CompositeType = If_t< useAssign, const ResultType, const DVecDVecMapExpr& >;

   //! Composite type of the left-hand side dense vector expression.
   using LeftOperand = If_t< IsExpression_v<VT1>, const VT1, const VT1& >;

   //! Composite type of the right-hand side dense vector expression.
   using RightOperand = If_t< IsExpression_v<VT2>, const VT2, const VT2& >;

   //! Data type of the custom binary operation.
   using Operation = OP;

   //! Type for the assignment of the left-hand side dense vector operand.
   using LT = If_t< RequiresEvaluation_v<VT1>, const RT1, CT1 >;

   //! Type for the assignment of the right-hand side dense vector operand.
   using RT = If_t< RequiresEvaluation_v<VT2>, const RT2, CT2 >;
   //**********************************************************************************************

   //**ConstIterator class definition**************************************************************
   /*!\brief Iterator over the elements of the dense vector map expression.
   */
   class ConstIterator
   {
    public:
      //**Type definitions*************************************************************************
      using IteratorCategory = std::random_access_iterator_tag;  //!< The iterator category.
      using ValueType        = ElementType;                      //!< Type of the underlying elements.
      using PointerType      = ElementType*;                     //!< Pointer return type.
      using ReferenceType    = ElementType&;                     //!< Reference return type.
      using DifferenceType   = ptrdiff_t;                        //!< Difference between two iterators.

      // STL iterator requirements
      using iterator_category = IteratorCategory;  //!< The iterator category.
      using value_type        = ValueType;         //!< Type of the underlying elements.
      using pointer           = PointerType;       //!< Pointer return type.
      using reference         = ReferenceType;     //!< Reference return type.
      using difference_type   = DifferenceType;    //!< Difference between two iterators.

      //! ConstIterator type of the left-hand side dense vector expression.
      using LeftIteratorType = ConstIterator_t<VT1>;

      //! ConstIterator type of the right-hand side dense vector expression.
      using RightIteratorType = ConstIterator_t<VT2>;
      //*******************************************************************************************

      //**Constructor******************************************************************************
      /*!\brief Constructor for the ConstIterator class.
      //
      // \param left Iterator to the initial left-hand side element.
      // \param right Iterator to the initial right-hand side element.
      // \param op The custom binary operation.
      */
      inline ConstIterator( LeftIteratorType left, RightIteratorType right, OP op )
         : left_ ( left  )          // Iterator to the current left-hand side element
         , right_( right )          // Iterator to the current right-hand side element
         , op_   ( std::move(op) )  // The custom binary operation
      {}
      //*******************************************************************************************

      //**Addition assignment operator*************************************************************
      /*!\brief Addition assignment operator.
      //
      // \param inc The increment of the iterator.
      // \return The incremented iterator.
      */
      inline BLAZE_DEVICE_CALLABLE ConstIterator& operator+=( size_t inc ) {
         left_  += inc;
         right_ += inc;
         return *this;
      }
      //*******************************************************************************************

      //**Subtraction assignment operator**********************************************************
      /*!\brief Subtraction assignment operator.
      //
      // \param dec The decrement of the iterator.
      // \return The decremented iterator.
      */
      inline BLAZE_DEVICE_CALLABLE ConstIterator& operator-=( size_t dec ) {
         left_  -= dec;
         right_ -= dec;
         return *this;
      }
      //*******************************************************************************************

      //**Prefix increment operator****************************************************************
      /*!\brief Pre-increment operator.
      //
      // \return Reference to the incremented iterator.
      */
      inline BLAZE_DEVICE_CALLABLE ConstIterator& operator++() {
         ++left_;
         ++right_;
         return *this;
      }
      //*******************************************************************************************

      //**Postfix increment operator***************************************************************
      /*!\brief Post-increment operator.
      //
      // \return The previous position of the iterator.
      */
      inline BLAZE_DEVICE_CALLABLE const ConstIterator operator++( int ) {
         return ConstIterator( left_++, right_++, op_ );
      }
      //*******************************************************************************************

      //**Prefix decrement operator****************************************************************
      /*!\brief Pre-decrement operator.
      //
      // \return Reference to the decremented iterator.
      */
      inline BLAZE_DEVICE_CALLABLE ConstIterator& operator--() {
         --left_;
         --right_;
         return *this;
      }
      //*******************************************************************************************

      //**Postfix decrement operator***************************************************************
      /*!\brief Post-decrement operator.
      //
      // \return The previous position of the iterator.
      */
      inline BLAZE_DEVICE_CALLABLE const ConstIterator operator--( int ) {
         return ConstIterator( left_--, right_--, op_ );
      }
      //*******************************************************************************************

      //**Element access operator******************************************************************
      /*!\brief Direct access to the element at the current iterator position.
      //
      // \return The resulting value.
      */
      inline ReturnType operator*() const {
         return op_( *left_, *right_ );
      }
      //*******************************************************************************************

      //**Load function****************************************************************************
      /*!\brief Access to the SIMD elements of the vector.
      //
      // \return The resulting SIMD element.
      */
      inline auto load() const noexcept {
         return op_.load( left_.load(), right_.load() );
      }
      //*******************************************************************************************

      //**Equality operator************************************************************************
      /*!\brief Equality comparison between two ConstIterator objects.
      //
      // \param rhs The right-hand side iterator.
      // \return \a true if the iterators refer to the same element, \a false if not.
      */
      inline bool operator==( const ConstIterator& rhs ) const {
         return left_ == rhs.left_;
      }
      //*******************************************************************************************

      //**Inequality operator**********************************************************************
      /*!\brief Inequality comparison between two ConstIterator objects.
      //
      // \param rhs The right-hand side iterator.
      // \return \a true if the iterators don't refer to the same element, \a false if they do.
      */
      inline bool operator!=( const ConstIterator& rhs ) const {
         return left_ != rhs.left_;
      }
      //*******************************************************************************************

      //**Less-than operator***********************************************************************
      /*!\brief Less-than comparison between two ConstIterator objects.
      //
      // \param rhs The right-hand side iterator.
      // \return \a true if the left-hand side iterator is smaller, \a false if not.
      */
      inline bool operator<( const ConstIterator& rhs ) const {
         return left_ < rhs.left_;
      }
      //*******************************************************************************************

      //**Greater-than operator********************************************************************
      /*!\brief Greater-than comparison between two ConstIterator objects.
      //
      // \param rhs The right-hand side iterator.
      // \return \a true if the left-hand side iterator is greater, \a false if not.
      */
      inline bool operator>( const ConstIterator& rhs ) const {
         return left_ > rhs.left_;
      }
      //*******************************************************************************************

      //**Less-or-equal-than operator**************************************************************
      /*!\brief Less-than comparison between two ConstIterator objects.
      //
      // \param rhs The right-hand side iterator.
      // \return \a true if the left-hand side iterator is smaller or equal, \a false if not.
      */
      inline bool operator<=( const ConstIterator& rhs ) const {
         return left_ <= rhs.left_;
      }
      //*******************************************************************************************

      //**Greater-or-equal-than operator***********************************************************
      /*!\brief Greater-than comparison between two ConstIterator objects.
      //
      // \param rhs The right-hand side iterator.
      // \return \a true if the left-hand side iterator is greater or equal, \a false if not.
      */
      inline bool operator>=( const ConstIterator& rhs ) const {
         return left_ >= rhs.left_;
      }
      //*******************************************************************************************

      //**Subtraction operator*********************************************************************
      /*!\brief Calculating the number of elements between two iterators.
      //
      // \param rhs The right-hand side iterator.
      // \return The number of elements between the two iterators.
      */
      inline DifferenceType operator-( const ConstIterator& rhs ) const {
         return left_ - rhs.left_;
      }
      //*******************************************************************************************

      //**Addition operator************************************************************************
      /*!\brief Addition between a ConstIterator and an integral value.
      //
      // \param it The iterator to be incremented.
      // \param inc The number of elements the iterator is incremented.
      // \return The incremented iterator.
      */
      friend inline const ConstIterator operator+( const ConstIterator& it, size_t inc ) {
         return ConstIterator( it.left_ + inc, it.right_ + inc, it.op_ );
      }
      //*******************************************************************************************

      //**Addition operator************************************************************************
      /*!\brief Addition between an integral value and a ConstIterator.
      //
      // \param inc The number of elements the iterator is incremented.
      // \param it The iterator to be incremented.
      // \return The incremented iterator.
      */
      friend inline const ConstIterator operator+( size_t inc, const ConstIterator& it ) {
         return ConstIterator( it.left_ + inc, it.right_ + inc, it.op_ );
      }
      //*******************************************************************************************

      //**Subtraction operator*********************************************************************
      /*!\brief Subtraction between a ConstIterator and an integral value.
      //
      // \param it The iterator to be decremented.
      // \param dec The number of elements the iterator is decremented.
      // \return The decremented iterator.
      */
      friend inline const ConstIterator operator-( const ConstIterator& it, size_t dec ) {
         return ConstIterator( it.left_ - dec, it.right_ - dec, it.op_ );
      }
      //*******************************************************************************************

    private:
      //**Member variables*************************************************************************
      LeftIteratorType  left_;   //!< Iterator to the current left-hand side element.
      RightIteratorType right_;  //!< Iterator to the current right-hand side element.
      OP                op_;     //!< The custom binary operation.
      //*******************************************************************************************
   };
   //**********************************************************************************************

   //**Compilation flags***************************************************************************
   //! Compilation switch for the expression template evaluation strategy.
   static constexpr bool simdEnabled =
      ( VT1::simdEnabled && VT2::simdEnabled &&
        If_t< HasSIMDEnabled_v<OP>, GetSIMDEnabled<OP,ET1,ET2>, HasLoad<OP> >::value );

   //! Compilation switch for the expression template assignment strategy.
   static constexpr bool smpAssignable = ( VT1::smpAssignable && VT2::smpAssignable );
   //**********************************************************************************************

   //**SIMD properties*****************************************************************************
   //! The number of elements packed within a single SIMD element.
   static constexpr size_t SIMDSIZE = SIMDTrait<ElementType>::size;
   //**********************************************************************************************

   //**Constructor*********************************************************************************
   /*!\brief Constructor for the DVecDVecMapExpr class.
   //
   // \param lhs The left-hand side dense vector operand of the map expression.
   // \param rhs The right-hand side dense vector operand of the map expression.
   // \param op The custom binary operation.
   */
   inline DVecDVecMapExpr( const VT1& lhs, const VT2& rhs, OP op ) noexcept
      : lhs_( lhs )            // Left-hand side dense vector of the map expression
      , rhs_( rhs )            // Right-hand side dense vector of the map expression
      , op_ ( std::move(op) )  // The custom binary operation
   {}
   //**********************************************************************************************

   //**Subscript operator**************************************************************************
   /*!\brief Subscript operator for the direct access to the vector elements.
   //
   // \param index Access index. The index has to be in the range \f$[0..N-1]\f$.
   // \return The resulting value.
   */
   inline ReturnType operator[]( size_t index ) const {
      BLAZE_INTERNAL_ASSERT( index < lhs_.size(), "Invalid vector access index" );
      return op_( lhs_[index], rhs_[index] );
   }
   //**********************************************************************************************

   //**At function*********************************************************************************
   /*!\brief Checked access to the vector elements.
   //
   // \param index Access index. The index has to be in the range \f$[0..N-1]\f$.
   // \return The resulting value.
   // \exception std::out_of_range Invalid vector access index.
   */
   inline ReturnType at( size_t index ) const {
      if( index >= lhs_.size() ) {
         BLAZE_THROW_OUT_OF_RANGE( "Invalid vector access index" );
      }
      return (*this)[index];
   }
   //**********************************************************************************************

   //**Load function*******************************************************************************
   /*!\brief Access to the SIMD elements of the vector.
   //
   // \param index Access index. The index has to be in the range \f$[0..N-1]\f$.
   // \return Reference to the accessed values.
   */
   BLAZE_ALWAYS_INLINE auto load( size_t index ) const noexcept {
      BLAZE_INTERNAL_ASSERT( index < lhs_.size()    , "Invalid vector access index" );
      BLAZE_INTERNAL_ASSERT( index % SIMDSIZE == 0UL, "Invalid vector access index" );
      return op_.load( lhs_.load( index ), rhs_.load( index ) );
   }
   //**********************************************************************************************

   //**Begin function******************************************************************************
   /*!\brief Returns an iterator to the first non-zero element of the dense vector.
   //
   // \return Iterator to the first non-zero element of the dense vector.
   */
   inline ConstIterator begin() const {
      return ConstIterator( lhs_.begin(), rhs_.begin(), op_ );
   }
   //**********************************************************************************************

   //**End function********************************************************************************
   /*!\brief Returns an iterator just past the last non-zero element of the dense vector.
   //
   // \return Iterator just past the last non-zero element of the dense vector.
   */
   inline ConstIterator end() const {
      return ConstIterator( lhs_.end(), rhs_.end(), op_ );
   }
   //**********************************************************************************************

   //**Size function*******************************************************************************
   /*!\brief Returns the current size/dimension of the vector.
   //
   // \return The size of the vector.
   */
   inline size_t size() const noexcept {
      return lhs_.size();
   }
   //**********************************************************************************************

   //**Left operand access*************************************************************************
   /*!\brief Returns the left-hand side dense vector operand.
   //
   // \return The left-hand side dense vector operand.
   */
   inline LeftOperand leftOperand() const noexcept {
      return lhs_;
   }
   //**********************************************************************************************

   //**Right operand access************************************************************************
   /*!\brief Returns the right-hand side dense vector operand.
   //
   // \return The right-hand side dense vector operand.
   */
   inline RightOperand rightOperand() const noexcept {
      return rhs_;
   }
   //**********************************************************************************************

   //**Operation access****************************************************************************
   /*!\brief Returns a copy of the custom operation.
   //
   // \return A copy of the custom operation.
   */
   inline Operation operation() const {
      return op_;
   }
   //**********************************************************************************************

   //**********************************************************************************************
   /*!\brief Returns whether the expression can alias with the given address \a alias.
   //
   // \param alias The alias to be checked.
   // \return \a true in case the expression can alias, \a false otherwise.
   */
   template< typename T >
   inline bool canAlias( const T* alias ) const noexcept {
      return ( IsExpression_v<VT1> && lhs_.canAlias( alias ) ) ||
             ( IsExpression_v<VT2> && rhs_.canAlias( alias ) );
   }
   //**********************************************************************************************

   //**********************************************************************************************
   /*!\brief Returns whether the expression is aliased with the given address \a alias.
   //
   // \param alias The alias to be checked.
   // \return \a true in case an alias effect is detected, \a false otherwise.
   */
   template< typename T >
   inline bool isAliased( const T* alias ) const noexcept {
      return ( lhs_.isAliased( alias ) || rhs_.isAliased( alias ) );
   }
   //**********************************************************************************************

   //**********************************************************************************************
   /*!\brief Returns whether the operands of the expression are properly aligned in memory.
   //
   // \return \a true in case the operands are aligned, \a false if not.
   */
   inline bool isAligned() const noexcept {
      return lhs_.isAligned() && rhs_.isAligned();
   }
   //**********************************************************************************************

   //**********************************************************************************************
   /*!\brief Returns whether the expression can be used in SMP assignments.
   //
   // \return \a true in case the expression can be used in SMP assignments, \a false if not.
   */
   inline bool canSMPAssign() const noexcept {
      return lhs_.canSMPAssign() && rhs_.canSMPAssign();
   }
   //**********************************************************************************************

 private:
   //**Member variables****************************************************************************
   LeftOperand  lhs_;  //!< Left-hand side dense vector of the map expression.
   RightOperand rhs_;  //!< Right-hand side dense vector of the map expression.
   Operation    op_;   //!< The custom binary operation.
   //**********************************************************************************************

   //**Assignment to dense vectors*****************************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief Assignment of a dense vector-dense vector map expression to a dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be assigned.
   // \return void
   //
   // This function implements the performance optimized assignment of a dense vector-dense
   // vector map expression to a dense vector. Due to the explicit application of the SFINAE
   // principle, this function can only be selected by the compiler in case either of the two
   // operands requires an intermediate evaluation.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto assign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( serial( rhs.lhs_ ) );  // Evaluation of the left-hand side dense vector operand
      RT y( serial( rhs.rhs_ ) );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      assign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**Assignment to sparse vectors****************************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief Assignment of a dense vector-dense vector map expression to a sparse vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side sparse vector.
   // \param rhs The right-hand side map expression to be assigned.
   // \return void
   //
   // This function implements the performance optimized assignment of a dense vector-dense
   // vector map expression to a sparse vector. Due to the explicit application of the SFINAE
   // principle, this function can only be selected by the compiler in case either of the two
   // operands requires an intermediate evaluation.
   */
   template< typename VT >  // Type of the target sparse vector
   friend inline auto assign( SparseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_CONSTRAINT_MUST_BE_DENSE_VECTOR_TYPE( ResultType );
      BLAZE_CONSTRAINT_MUST_BE_VECTOR_WITH_TRANSPOSE_FLAG( ResultType, TF );
      BLAZE_CONSTRAINT_MUST_NOT_REQUIRE_EVALUATION( ResultType );

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      const ResultType tmp( serial( rhs ) );
      assign( *lhs, tmp );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**Addition assignment to dense vectors********************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief Addition assignment of a dense vector-dense vector map expression to a dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be added.
   // \return void
   //
   // This function implements the performance optimized addition assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application
   // of the SFINAE principle, this function can only be selected by the compiler in case
   // either of the two operands requires an intermediate evaluation.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto addAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( serial( rhs.lhs_ ) );  // Evaluation of the left-hand side dense vector operand
      RT y( serial( rhs.rhs_ ) );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      addAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**Addition assignment to sparse vectors*******************************************************
   // No special implementation for the addition assignment to sparse vectors.
   //**********************************************************************************************

   //**Subtraction assignment to dense vectors*****************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief Subtraction assignment of a dense vector-dense vector map expression to a dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be subtracted.
   // \return void
   //
   // This function implements the performance optimized subtraction assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application
   // of the SFINAE principle, this function can only be selected by the compiler in case
   // either of the two operands requires an intermediate evaluation.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto subAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( serial( rhs.lhs_ ) );  // Evaluation of the left-hand side dense vector operand
      RT y( serial( rhs.rhs_ ) );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      subAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**Subtraction assignment to sparse vectors****************************************************
   // No special implementation for the subtraction assignment to sparse vectors.
   //**********************************************************************************************

   //**Multiplication assignment to dense vectors**************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief Multiplication assignment of a dense vector-dense vector map expression to a dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be multiplied.
   // \return void
   //
   // This function implements the performance optimized multiplication assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application of
   // the SFINAE principle, this function can only be selected by the compiler in case either
   // of the two operands requires an intermediate evaluation.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto multAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( serial( rhs.lhs_ ) );  // Evaluation of the left-hand side dense vector operand
      RT y( serial( rhs.rhs_ ) );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      multAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**Multiplication assignment to sparse vectors*************************************************
   // No special implementation for the multiplication assignment to sparse vectors.
   //**********************************************************************************************

   //**Division assignment to dense vectors********************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief Division assignment of a dense vector-dense vector map expression to a dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression divisor.
   // \return void
   //
   // This function implements the performance optimized division assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application
   // of the SFINAE principle, this function can only be selected by the compiler in case
   // either of the two operands requires an intermediate evaluation.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto divAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( serial( rhs.lhs_ ) );  // Evaluation of the left-hand side dense vector operand
      RT y( serial( rhs.rhs_ ) );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      divAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**Division assignment to sparse vectors*******************************************************
   // No special implementation for the division assignment to sparse vectors.
   //**********************************************************************************************

   //**SMP assignment to dense vectors*************************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief SMP assignment of a dense vector-dense vector map expression to a dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be assigned.
   // \return void
   //
   // This function implements the performance optimized SMP assignment of a dense vector-dense
   // vector map expression to a dense vector. Due to the explicit application of the SFINAE
   // principle, this function can only be selected by the compiler in case the expression
   // specific parallel evaluation strategy is selected.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto smpAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseSMPAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( rhs.lhs_ );  // Evaluation of the left-hand side dense vector operand
      RT y( rhs.rhs_ );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      smpAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**SMP assignment to sparse vectors************************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief SMP assignment of a dense vector-dense vector map expression to a sparse vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side sparse vector.
   // \param rhs The right-hand side map expression to be assigned.
   // \return void
   //
   // This function implements the performance optimized SMP assignment of a dense vector-dense
   // vector map expression to a sparse vector. Due to the explicit application of the SFINAE
   // principle, this function can only be selected by the compiler in case the expression
   // specific parallel evaluation strategy is selected.
   */
   template< typename VT >  // Type of the target sparse vector
   friend inline auto smpAssign( SparseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseSMPAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_CONSTRAINT_MUST_BE_DENSE_VECTOR_TYPE( ResultType );
      BLAZE_CONSTRAINT_MUST_BE_VECTOR_WITH_TRANSPOSE_FLAG( ResultType, TF );
      BLAZE_CONSTRAINT_MUST_NOT_REQUIRE_EVALUATION( ResultType );

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      const ResultType tmp( rhs );
      smpAssign( *lhs, tmp );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**SMP addition assignment to dense vectors****************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief SMP addition assignment of a dense vector-dense vector map expression to a dense
   //        vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be added.
   // \return void
   //
   // This function implements the performance optimized SMP addition assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application
   // of the SFINAE principle, this function can only be selected by the compiler in case the
   // expression specific parallel evaluation strategy is selected.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto smpAddAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseSMPAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( rhs.lhs_ );  // Evaluation of the left-hand side dense vector operand
      RT y( rhs.rhs_ );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      smpAddAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**SMP addition assignment to sparse vectors***************************************************
   // No special implementation for the SMP addition assignment to sparse vectors.
   //**********************************************************************************************

   //**SMP subtraction assignment to dense vectors*************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief SMP subtraction assignment of a dense vector-dense vector map expression to a dense
   //        vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be subtracted.
   // \return void
   //
   // This function implements the performance optimized SMP subtraction assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application of
   // the SFINAE principle, this function can only be selected by the compiler in case the
   // expression specific parallel evaluation strategy is selected.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto smpSubAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseSMPAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( rhs.lhs_ );  // Evaluation of the left-hand side dense vector operand
      RT y( rhs.rhs_ );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      smpSubAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**SMP subtraction assignment to sparse vectors************************************************
   // No special implementation for the SMP subtraction assignment to sparse vectors.
   //**********************************************************************************************

   //**SMP multiplication assignment to dense vectors**********************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief SMP multiplication assignment of a dense vector-dense vector map expression to a
   //        dense vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression to be multiplied.
   // \return void
   //
   // This function implements the performance optimized SMP multiplication assignment of a
   // dense vector-dense vector map expression to a dense vector. Due to the explicit application
   // of the SFINAE principle, this function can only be selected by the compiler in case the
   // expression specific parallel evaluation strategy is selected.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto smpMultAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseSMPAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( rhs.lhs_ );  // Evaluation of the left-hand side dense vector operand
      RT y( rhs.rhs_ );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      smpMultAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**SMP multiplication assignment to sparse vectors*********************************************
   // No special implementation for the SMP multiplication assignment to sparse vectors.
   //**********************************************************************************************

   //**SMP division assignment to dense vectors****************************************************
   /*! \cond BLAZE_INTERNAL */
   /*!\brief SMP division assignment of a dense vector-dense vector map expression to a dense
   //        vector.
   // \ingroup dense_vector
   //
   // \param lhs The target left-hand side dense vector.
   // \param rhs The right-hand side map expression divisor.
   // \return void
   //
   // This function implements the performance optimized SMP division assignment of a dense
   // vector-dense vector map expression to a dense vector. Due to the explicit application
   // of the SFINAE principle, this function can only be selected by the compiler in case
   // the expression specific parallel evaluation strategy is selected.
   */
   template< typename VT >  // Type of the target dense vector
   friend inline auto smpDivAssign( DenseVector<VT,TF>& lhs, const DVecDVecMapExpr& rhs )
      -> EnableIf_t< UseSMPAssign_v<VT> >
   {
      BLAZE_FUNCTION_TRACE;

      BLAZE_INTERNAL_ASSERT( (*lhs).size() == rhs.size(), "Invalid vector sizes" );

      LT x( rhs.lhs_ );  // Evaluation of the left-hand side dense vector operand
      RT y( rhs.rhs_ );  // Evaluation of the right-hand side dense vector operand

      BLAZE_INTERNAL_ASSERT( x.size() == rhs.lhs_.size() , "Invalid vector size" );
      BLAZE_INTERNAL_ASSERT( y.size() == rhs.rhs_.size() , "Invalid vector size" );

      smpDivAssign( *lhs, map( x, y, rhs.op_ ) );
   }
   /*! \endcond */
   //**********************************************************************************************

   //**SMP division assignment to sparse vectors***************************************************
   // No special implementation for the SMP division assignment to sparse vectors.
   //**********************************************************************************************

   //**Compile time checks*************************************************************************
   /*! \cond BLAZE_INTERNAL */
   BLAZE_CONSTRAINT_MUST_BE_DENSE_VECTOR_TYPE( VT1 );
   BLAZE_CONSTRAINT_MUST_BE_DENSE_VECTOR_TYPE( VT2 );
   BLAZE_CONSTRAINT_MUST_BE_VECTOR_WITH_TRANSPOSE_FLAG( VT1, TF );
   BLAZE_CONSTRAINT_MUST_BE_VECTOR_WITH_TRANSPOSE_FLAG( VT2, TF );
   /*! \endcond */
   //**********************************************************************************************
};
//*************************************************************************************************




//=================================================================================================
//
//  GLOBAL FUNCTIONS
//
//=================================================================================================

//*************************************************************************************************
/*!\brief Elementwise evaluation of the given binary operation on each single element of the
//        dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector operand.
// \param rhs The right-hand side dense vector operand.
// \param op The custom, binary operation.
// \return The binary operation applied to each single element of \a lhs and \a rhs.
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a map() function evaluates the given binary operation on each single element of the input
// vectors \a lhs and \a rhs. The function returns an expression representing this operation.\n
// The following example demonstrates the use of the \a map() function:

   \code
   blaze::DynamicVector<double> a, b, c;
   // ... Resizing and initialization
   c = map( a, b, []( double x, double y ){ return std::min( x, y ); } );
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1   // Type of the left-hand side dense vector
        , typename VT2   // Type of the right-hand side dense vector
        , bool TF        // Transpose flag
        , typename OP >  // Type of the custom operation
inline decltype(auto)
   map( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs, OP op )
{
   BLAZE_FUNCTION_TRACE;

   if( (*lhs).size() != (*rhs).size() ) {
      BLAZE_THROW_INVALID_ARGUMENT( "Vector sizes do not match" );
   }

   using ReturnType = const DVecDVecMapExpr<VT1,VT2,OP,TF>;
   return ReturnType( *lhs, *rhs, std::move(op) );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Elementwise evaluation of the given ternary operation on each single element of the
//        dense vectors \a dv1, \a dv2, and \a dv3.
// \ingroup dense_vector
//
// \param dv1 The first dense vector operand.
// \param dv2 The second dense vector operand.
// \param dv3 The third dense vector operand.
// \param op The custom, ternary operation.
// \return The ternary operation applied to each single element of the three vectors.
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a map() function evaluates the given ternary operation on each single element of the
// input vectors \a dv1, \a dv2, and \a dv3. The function returns an expression representing
// this operation.\n
// In case the current sizes of the three given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1   // Type of the first dense vector
        , typename VT2   // Type of the second dense vector
        , typename VT3   // Type of the third dense vector
        , bool TF        // Transpose flag
        , typename OP >  // Type of the custom operation
inline decltype(auto)
   map( const DenseVector<VT1,TF>& dv1, const DenseVector<VT2,TF>& dv2,
        const DenseVector<VT3,TF>& dv3, OP op )
{
   BLAZE_FUNCTION_TRACE;

   const MakePair mp{};
   return map( map( map( *dv1, *dv2, mp ), *dv3, mp ), join( std::move(op) ) );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Elementwise evaluation of the given 4-ary operation on each single element of the
//        dense vectors \a dv1, \a dv2, \a dv3, and \a dv4.
// \ingroup dense_vector
//
// \param dv1 The first dense vector operand.
// \param dv2 The second dense vector operand.
// \param dv3 The third dense vector operand.
// \param dv4 The fourth dense vector operand.
// \param op The custom, 4-ary operation.
// \return The 4-ary operation applied to each single element of the four vectors.
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a map() function evaluates the given 4-ary operation on each single element of the input
// vectors \a dv1, \a dv2, \a dv3, and \a dv4. The function returns an expression representing
// this operation.\n
// In case the current sizes of the four given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1   // Type of the first dense vector
        , typename VT2   // Type of the second dense vector
        , typename VT3   // Type of the third dense vector
        , typename VT4   // Type of the fourth dense vector
        , bool TF        // Transpose flag
        , typename OP >  // Type of the custom operation
inline decltype(auto)
   map( const DenseVector<VT1,TF>& dv1, const DenseVector<VT2,TF>& dv2,
        const DenseVector<VT3,TF>& dv3, const DenseVector<VT4,TF>& dv4, OP op )
{
   BLAZE_FUNCTION_TRACE;

   const MakePair mp{};
   return map( map( map( map( *dv1, *dv2, mp ), *dv3, mp ), *dv4, mp ), join( std::move(op) ) );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Elementwise evaluation of the given 5-ary operation on each single element of the
//        dense vectors \a dv1, \a dv2, \a dv3, \a dv4, and \a dv5.
// \ingroup dense_vector
//
// \param dv1 The first dense vector operand.
// \param dv2 The second dense vector operand.
// \param dv3 The third dense vector operand.
// \param dv4 The fourth dense vector operand.
// \param dv5 The fifth dense vector operand.
// \param op The custom, 5-ary operation.
// \return The 5-ary operation applied to each single element of the five vectors.
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a map() function evaluates the given 5-ary operation on each single element of the input
// vectors \a dv1, \a dv2, \a dv3, \a dv4, and \a dv5. The function returns an expression
// representing this operation.\n
// In case the current sizes of the five given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1   // Type of the first dense vector
        , typename VT2   // Type of the second dense vector
        , typename VT3   // Type of the third dense vector
        , typename VT4   // Type of the fourth dense vector
        , typename VT5   // Type of the fifth dense vector
        , bool TF        // Transpose flag
        , typename OP >  // Type of the custom operation
inline decltype(auto)
   map( const DenseVector<VT1,TF>& dv1, const DenseVector<VT2,TF>& dv2,
        const DenseVector<VT3,TF>& dv3, const DenseVector<VT4,TF>& dv4,
        const DenseVector<VT5,TF>& dv5, OP op )
{
   BLAZE_FUNCTION_TRACE;

   const MakePair mp{};
   return map( map( map( map( map( *dv1, *dv2, mp ), *dv3, mp ), *dv4, mp ), *dv5, mp ), join( std::move(op) ) );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Elementwise evaluation of the given 6-ary operation on each single element of the
//        dense vectors \a dv1, \a dv2, \a dv3, \a dv4, \a dv5, and \a dv6.
// \ingroup dense_vector
//
// \param dv1 The first dense vector operand.
// \param dv2 The second dense vector operand.
// \param dv3 The third dense vector operand.
// \param dv4 The fourth dense vector operand.
// \param dv5 The fifth dense vector operand.
// \param dv6 The sixth dense vector operand.
// \param op The custom, 6-ary operation.
// \return The 6-ary operation applied to each single element of the six vectors.
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a map() function evaluates the given 6-ary operation on each single element of the
// input vectors \a dv1, \a dv2, \a dv3, \a dv4, \a dv5, and \a dv6. The function returns an
// expression representing this operation.\n
// In case the current sizes of the six given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1   // Type of the first dense vector
        , typename VT2   // Type of the second dense vector
        , typename VT3   // Type of the third dense vector
        , typename VT4   // Type of the fourth dense vector
        , typename VT5   // Type of the fifth dense vector
        , typename VT6   // Type of the sixth dense vector
        , bool TF        // Transpose flag
        , typename OP >  // Type of the custom operation
inline decltype(auto)
   map( const DenseVector<VT1,TF>& dv1, const DenseVector<VT2,TF>& dv2,
        const DenseVector<VT3,TF>& dv3, const DenseVector<VT4,TF>& dv4,
        const DenseVector<VT5,TF>& dv5, const DenseVector<VT6,TF>& dv6, OP op )
{
   BLAZE_FUNCTION_TRACE;

   const MakePair mp{};
   return map( map( map( map( map( map( *dv1, *dv2, mp ), *dv3, mp ), *dv4, mp ), *dv5, mp ), *dv6, mp ), join( std::move(op) ) );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Computes the componentwise minimum of the dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector operand.
// \param rhs The right-hand side dense vector operand.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This function computes the componentwise minimum of the two dense vectors \a lhs and \a rhs.
// The function returns an expression representing this operation.\n
// The following example demonstrates the use of the \a min() function:

   \code
   blaze::DynamicVector<double> a, b, c;
   // ... Resizing and initialization
   c = min( a, b );
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   min( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Min() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Computes the componentwise maximum of the dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector operand.
// \param rhs The right-hand side dense vector operand.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This function computes the componentwise maximum of the two dense vectors \a lhs and \a rhs.
// The function returns an expression representing this operation.\n
// The following example demonstrates the use of the \a max() function:

   \code
   blaze::DynamicVector<double> a, b, c;
   // ... Resizing and initialization
   c = max( a, b );
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   max( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Max() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Computes the componentwise hypotenous for the dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector operand.
// \param rhs The right-hand side dense vector operand.
// \return The resulting dense vector
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a hypot() function computes the componentwise hypotenous for the two dense vectors
// \a lhs and \a rhs. The function returns an expression representing this operation.\n
// The following example demonstrates the use of the \a hypot() function:

   \code
   blaze::DynamicVector<double> a, b, c;
   // ... Resizing and initialization
   c = hypot( a, b );
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   hypot( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Hypot() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Computes the componentwise exponential value for the dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector operand.
// \param rhs The right-hand side dense vector operand.
// \return The resulting dense vector
// \exception std::invalid_argument Vector sizes do not match.
//
// The \a pow() function computes the componentwise exponential value for the two dense vectors
// \a lhs and \a rhs. The function returns an expression representing this operation.\n
// The following example demonstrates the use of the \a pow() function:

   \code
   blaze::DynamicVector<double> a, b, c;
   // ... Resizing and initialization
   c = pow( a, b );
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   pow( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Pow() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Computes the multi-valued inverse tangent of the dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector operand.
// \param rhs The right-hand side dense vector operand.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This function computes the multi-valued inverse tangent of the two dense vectors \a lhs and
// \a rhs. The function returns an expression representing this operation.\n
// The following example demonstrates the use of the \a atan2() function:

   \code
   blaze::DynamicVector<double> a, b, c;
   // ... Resizing and initialization
   c = atan2( a, b );
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   atan2( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Atan2() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Elementwise conditional selection of values from the dense vectors \a lhs and \a rhs.
// \ingroup dense_vector
//
// \param cond The dense vector containing the selection conditions.
// \param lhs The true-case dense vector.
// \param rhs The false-case dense vector.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This function performs an elementwise conditional selection of values from the two given dense
// vectors \a lhs and \a rhs. In case an element in the \a cond vector evaluates to \a true, the
// according element of \a lhs is selected, in case the \a cond element evaluates to \a false, the
// according element of \a rhs is selected. The function returns an expression representing this
// operation.\n
// The following example demonstrates the use of the \a selec() function:

   \code
   blaze::DynamicVector<bool> cond{ true, false, true false };
   blaze::DynamicVector<int> a{ 1, -1, 1, -1 };
   blaze::DynamicVector<int> b{ -2, 2, -2, 2 };
   blaze::DynamicVector<int> c;
   // ... Resizing and initialization

   c = select( cond, a, b );  // Results in ( 1, 2, 1, 2 )
   \endcode

// In case the current sizes of the three given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the conditional dense vector
        , typename VT2  // Type of the true-case dense vector
        , typename VT3  // Type of the false-case dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   select( const DenseVector<VT1,TF>& cond, const DenseVector<VT2,TF>& lhs, const DenseVector<VT3,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( cond, lhs, rhs, []( bool c, const auto& a, const auto& b ) { return c ? a : b; } );
}
//*************************************************************************************************




//=================================================================================================
//
//  GLOBAL BINARY ARITHMETIC OPERATORS
//
//=================================================================================================

//*************************************************************************************************
/*!\brief Left-shift operator for the elementwise left-shift of a dense vector.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector to be shifted.
// \param rhs The right-hand side dense vector of bits to shift.
// \return The left-shifted dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the elementwise left-shift of a given dense vector:

   \code
   blaze::DynamicVector<unsigned int> a, b, c;
   // ... Resizing and initialization
   c = a << b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator<<( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, ShiftLV() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Right-shift operator for the elementwise right-shift of a dense vector.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector to be shifted.
// \param rhs The right-hand side dense vector of bits to shift.
// \return The right-shifted dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the elementwise right-shift of a given dense vector:

   \code
   blaze::DynamicVector<unsigned int> a, b, c;
   // ... Resizing and initialization
   c = a >> b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator>>( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, ShiftRV() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Bitwise AND operator for two dense vectors.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector for the bitwise AND operation.
// \param rhs The right-hand side dense vector for the bitwise AND operation.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the bitwise AND of the given two dense vectors:

   \code
   blaze::DynamicVector<unsigned int> a, b, c;
   // ... Resizing and initialization
   c = a & b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator&( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Bitand() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Bitwise OR operator for two dense vectors.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector for the bitwise OR operation.
// \param rhs The right-hand side dense vector for the bitwise OR operation.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the bitwise OR of the given two dense vectors:

   \code
   blaze::DynamicVector<unsigned int> a, b, c;
   // ... Resizing and initialization
   c = a | b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator|( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Bitor() );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Bitwise XOR operator for two dense vectors.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector for the bitwise XOR operation.
// \param rhs The right-hand side dense vector for the bitwise XOR operation.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the bitwise XOR of the given two dense vectors:

   \code
   blaze::DynamicVector<unsigned int> a, b, c;
   // ... Resizing and initialization
   c = a ^ b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator^( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Bitxor() );
}
//*************************************************************************************************




//=================================================================================================
//
//  GLOBAL RESTRUCTURING BINARY ARITHMETIC OPERATORS
//
//=================================================================================================

//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Left-shift operator for the elementwise left-shift of an elementwise left-shift expression
//        (\f$ \vec{a}=\vec{b}\ll\vec{c}\ll\vec{d} \f$).
// \ingroup dense_vector
//
// \param lhs The left-hand side left-shift expression to be shifted.
// \param rhs The right-hand side dense vector of bits to shift.
// \return The left-shifted dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This function implements a performance optimized treatment of the elementwise left-shift
// operation on an elementwise left-shift expression.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the middle dense vector
        , typename VT3  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator<<( const DVecDVecMapExpr<VT1,VT2,ShiftLV,TF>& lhs, const DenseVector<VT3,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( lhs.leftOperand(), lhs.rightOperand() + (*rhs), lhs.operation() );
}
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
/*!\brief Right-shift operator for the elementwise right-shift of an elementwise right-shift
//        expression (\f$ \vec{a}=\vec{b}\gg\vec{c}\gg\vec{d} \f$).
// \ingroup dense_vector
//
// \param lhs The left-hand side right-shift expression to be shifted.
// \param rhs The right-hand side dense vector of bits to shift.
// \return The right-shifted dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This function implements a performance optimized treatment of the elementwise right-shift
// operation on an elementwise right-shift expression.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the middle dense vector
        , typename VT3  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator>>( const DVecDVecMapExpr<VT1,VT2,ShiftRV,TF>& lhs, const DenseVector<VT3,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( lhs.leftOperand(), lhs.rightOperand() + (*rhs), lhs.operation() );
}
/*! \endcond */
//*************************************************************************************************




//=================================================================================================
//
//  GLOBAL BINARY LOGICAL OPERATORS
//
//=================================================================================================

//*************************************************************************************************
/*!\brief Logical AND operator for two dense vectors.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector for the logical AND operation.
// \param rhs The right-hand side dense vector for the logical AND operation.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the logical AND of the given two dense vectors:

   \code
   blaze::DynamicVector<bool> a, b, c;
   // ... Resizing and initialization
   c = a && b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator&&( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, And{} );
}
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Logical OR operator for two dense vectors.
// \ingroup dense_vector
//
// \param lhs The left-hand side dense vector for the logical OR operation.
// \param rhs The right-hand side dense vector for the logical OR operation.
// \return The resulting dense vector.
// \exception std::invalid_argument Vector sizes do not match.
//
// This operator represents the logical OR of the given two dense vectors:

   \code
   blaze::DynamicVector<bool> a, b, c;
   // ... Resizing and initialization
   c = a || b;
   \endcode

// In case the current sizes of the two given vectors don't match, a \a std::invalid_argument
// is thrown.
*/
template< typename VT1  // Type of the left-hand side dense vector
        , typename VT2  // Type of the right-hand side dense vector
        , bool TF >     // Transpose flag
inline decltype(auto)
   operator||( const DenseVector<VT1,TF>& lhs, const DenseVector<VT2,TF>& rhs )
{
   BLAZE_FUNCTION_TRACE;

   return map( *lhs, *rhs, Or{} );
}
//*************************************************************************************************




//=================================================================================================
//
//  ISALIGNED SPECIALIZATIONS
//
//=================================================================================================

//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
template< typename VT1, typename VT2, typename OP, bool TF >
struct IsAligned< DVecDVecMapExpr<VT1,VT2,OP,TF> >
   : public BoolConstant< IsAligned_v<VT1> && IsAligned_v<VT2> >
{};
/*! \endcond */
//*************************************************************************************************




//=================================================================================================
//
//  ISPADDED SPECIALIZATIONS
//
//=================================================================================================

//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
template< typename VT1, typename VT2, typename OP, bool TF >
struct IsPadded< DVecDVecMapExpr<VT1,VT2,OP,TF> >
   : public BoolConstant< IsPadded_v<VT1> && IsPadded_v<VT2> && IsPaddingEnabled_v<OP> >
{};
/*! \endcond */
//*************************************************************************************************

} // namespace blaze

#endif
