﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2Request.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancingv2 {
namespace Model {

/**
 */
class DescribeTargetGroupsRequest : public ElasticLoadBalancingv2Request {
 public:
  AWS_ELASTICLOADBALANCINGV2_API DescribeTargetGroupsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeTargetGroups"; }

  AWS_ELASTICLOADBALANCINGV2_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCINGV2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the load balancer.</p>
   */
  inline const Aws::String& GetLoadBalancerArn() const { return m_loadBalancerArn; }
  inline bool LoadBalancerArnHasBeenSet() const { return m_loadBalancerArnHasBeenSet; }
  template <typename LoadBalancerArnT = Aws::String>
  void SetLoadBalancerArn(LoadBalancerArnT&& value) {
    m_loadBalancerArnHasBeenSet = true;
    m_loadBalancerArn = std::forward<LoadBalancerArnT>(value);
  }
  template <typename LoadBalancerArnT = Aws::String>
  DescribeTargetGroupsRequest& WithLoadBalancerArn(LoadBalancerArnT&& value) {
    SetLoadBalancerArn(std::forward<LoadBalancerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Names (ARN) of the target groups.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTargetGroupArns() const { return m_targetGroupArns; }
  inline bool TargetGroupArnsHasBeenSet() const { return m_targetGroupArnsHasBeenSet; }
  template <typename TargetGroupArnsT = Aws::Vector<Aws::String>>
  void SetTargetGroupArns(TargetGroupArnsT&& value) {
    m_targetGroupArnsHasBeenSet = true;
    m_targetGroupArns = std::forward<TargetGroupArnsT>(value);
  }
  template <typename TargetGroupArnsT = Aws::Vector<Aws::String>>
  DescribeTargetGroupsRequest& WithTargetGroupArns(TargetGroupArnsT&& value) {
    SetTargetGroupArns(std::forward<TargetGroupArnsT>(value));
    return *this;
  }
  template <typename TargetGroupArnsT = Aws::String>
  DescribeTargetGroupsRequest& AddTargetGroupArns(TargetGroupArnsT&& value) {
    m_targetGroupArnsHasBeenSet = true;
    m_targetGroupArns.emplace_back(std::forward<TargetGroupArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The names of the target groups.</p>
   */
  inline const Aws::Vector<Aws::String>& GetNames() const { return m_names; }
  inline bool NamesHasBeenSet() const { return m_namesHasBeenSet; }
  template <typename NamesT = Aws::Vector<Aws::String>>
  void SetNames(NamesT&& value) {
    m_namesHasBeenSet = true;
    m_names = std::forward<NamesT>(value);
  }
  template <typename NamesT = Aws::Vector<Aws::String>>
  DescribeTargetGroupsRequest& WithNames(NamesT&& value) {
    SetNames(std::forward<NamesT>(value));
    return *this;
  }
  template <typename NamesT = Aws::String>
  DescribeTargetGroupsRequest& AddNames(NamesT&& value) {
    m_namesHasBeenSet = true;
    m_names.emplace_back(std::forward<NamesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The marker for the next set of results. (You received this marker from a
   * previous call.)</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeTargetGroupsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with this call.</p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline DescribeTargetGroupsRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_loadBalancerArn;

  Aws::Vector<Aws::String> m_targetGroupArns;

  Aws::Vector<Aws::String> m_names;

  Aws::String m_marker;

  int m_pageSize{0};
  bool m_loadBalancerArnHasBeenSet = false;
  bool m_targetGroupArnsHasBeenSet = false;
  bool m_namesHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancingv2
}  // namespace Aws
