﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/signer/SignerRequest.h>
#include <aws/signer/Signer_EXPORTS.h>
#include <aws/signer/model/Destination.h>
#include <aws/signer/model/Source.h>

#include <utility>

namespace Aws {
namespace signer {
namespace Model {

/**
 */
class StartSigningJobRequest : public SignerRequest {
 public:
  AWS_SIGNER_API StartSigningJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartSigningJob"; }

  AWS_SIGNER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The S3 bucket that contains the object to sign or a BLOB that contains your
   * raw code.</p>
   */
  inline const Source& GetSource() const { return m_source; }
  inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
  template <typename SourceT = Source>
  void SetSource(SourceT&& value) {
    m_sourceHasBeenSet = true;
    m_source = std::forward<SourceT>(value);
  }
  template <typename SourceT = Source>
  StartSigningJobRequest& WithSource(SourceT&& value) {
    SetSource(std::forward<SourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The S3 bucket in which to save your signed object. The destination contains
   * the name of your bucket and an optional prefix.</p>
   */
  inline const Destination& GetDestination() const { return m_destination; }
  inline bool DestinationHasBeenSet() const { return m_destinationHasBeenSet; }
  template <typename DestinationT = Destination>
  void SetDestination(DestinationT&& value) {
    m_destinationHasBeenSet = true;
    m_destination = std::forward<DestinationT>(value);
  }
  template <typename DestinationT = Destination>
  StartSigningJobRequest& WithDestination(DestinationT&& value) {
    SetDestination(std::forward<DestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the signing profile.</p>
   */
  inline const Aws::String& GetProfileName() const { return m_profileName; }
  inline bool ProfileNameHasBeenSet() const { return m_profileNameHasBeenSet; }
  template <typename ProfileNameT = Aws::String>
  void SetProfileName(ProfileNameT&& value) {
    m_profileNameHasBeenSet = true;
    m_profileName = std::forward<ProfileNameT>(value);
  }
  template <typename ProfileNameT = Aws::String>
  StartSigningJobRequest& WithProfileName(ProfileNameT&& value) {
    SetProfileName(std::forward<ProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>String that identifies the signing request. All calls after the first that
   * use this token return the same response as the first call.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  StartSigningJobRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The AWS account ID of the signing profile owner.</p>
   */
  inline const Aws::String& GetProfileOwner() const { return m_profileOwner; }
  inline bool ProfileOwnerHasBeenSet() const { return m_profileOwnerHasBeenSet; }
  template <typename ProfileOwnerT = Aws::String>
  void SetProfileOwner(ProfileOwnerT&& value) {
    m_profileOwnerHasBeenSet = true;
    m_profileOwner = std::forward<ProfileOwnerT>(value);
  }
  template <typename ProfileOwnerT = Aws::String>
  StartSigningJobRequest& WithProfileOwner(ProfileOwnerT&& value) {
    SetProfileOwner(std::forward<ProfileOwnerT>(value));
    return *this;
  }
  ///@}
 private:
  Source m_source;

  Destination m_destination;

  Aws::String m_profileName;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_profileOwner;
  bool m_sourceHasBeenSet = false;
  bool m_destinationHasBeenSet = false;
  bool m_profileNameHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = true;
  bool m_profileOwnerHasBeenSet = false;
};

}  // namespace Model
}  // namespace signer
}  // namespace Aws
