﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/InferenceExperimentStatus.h>
#include <aws/sagemaker/model/InferenceExperimentType.h>
#include <aws/sagemaker/model/SortInferenceExperimentsBy.h>
#include <aws/sagemaker/model/SortOrder.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class ListInferenceExperimentsRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API ListInferenceExperimentsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListInferenceExperiments"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Selects inference experiments whose names contain this name.</p>
   */
  inline const Aws::String& GetNameContains() const { return m_nameContains; }
  inline bool NameContainsHasBeenSet() const { return m_nameContainsHasBeenSet; }
  template <typename NameContainsT = Aws::String>
  void SetNameContains(NameContainsT&& value) {
    m_nameContainsHasBeenSet = true;
    m_nameContains = std::forward<NameContainsT>(value);
  }
  template <typename NameContainsT = Aws::String>
  ListInferenceExperimentsRequest& WithNameContains(NameContainsT&& value) {
    SetNameContains(std::forward<NameContainsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Selects inference experiments of this type. For the possible types of
   * inference experiments, see <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateInferenceExperiment.html">CreateInferenceExperiment</a>.
   * </p>
   */
  inline InferenceExperimentType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(InferenceExperimentType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline ListInferenceExperimentsRequest& WithType(InferenceExperimentType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Selects inference experiments which are in this status. For the possible
   * statuses, see <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeInferenceExperiment.html">DescribeInferenceExperiment</a>.
   * </p>
   */
  inline InferenceExperimentStatus GetStatusEquals() const { return m_statusEquals; }
  inline bool StatusEqualsHasBeenSet() const { return m_statusEqualsHasBeenSet; }
  inline void SetStatusEquals(InferenceExperimentStatus value) {
    m_statusEqualsHasBeenSet = true;
    m_statusEquals = value;
  }
  inline ListInferenceExperimentsRequest& WithStatusEquals(InferenceExperimentStatus value) {
    SetStatusEquals(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Selects inference experiments which were created after this timestamp.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeAfter() const { return m_creationTimeAfter; }
  inline bool CreationTimeAfterHasBeenSet() const { return m_creationTimeAfterHasBeenSet; }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  void SetCreationTimeAfter(CreationTimeAfterT&& value) {
    m_creationTimeAfterHasBeenSet = true;
    m_creationTimeAfter = std::forward<CreationTimeAfterT>(value);
  }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  ListInferenceExperimentsRequest& WithCreationTimeAfter(CreationTimeAfterT&& value) {
    SetCreationTimeAfter(std::forward<CreationTimeAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Selects inference experiments which were created before this timestamp.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeBefore() const { return m_creationTimeBefore; }
  inline bool CreationTimeBeforeHasBeenSet() const { return m_creationTimeBeforeHasBeenSet; }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  void SetCreationTimeBefore(CreationTimeBeforeT&& value) {
    m_creationTimeBeforeHasBeenSet = true;
    m_creationTimeBefore = std::forward<CreationTimeBeforeT>(value);
  }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  ListInferenceExperimentsRequest& WithCreationTimeBefore(CreationTimeBeforeT&& value) {
    SetCreationTimeBefore(std::forward<CreationTimeBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Selects inference experiments which were last modified after this
   * timestamp.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTimeAfter() const { return m_lastModifiedTimeAfter; }
  inline bool LastModifiedTimeAfterHasBeenSet() const { return m_lastModifiedTimeAfterHasBeenSet; }
  template <typename LastModifiedTimeAfterT = Aws::Utils::DateTime>
  void SetLastModifiedTimeAfter(LastModifiedTimeAfterT&& value) {
    m_lastModifiedTimeAfterHasBeenSet = true;
    m_lastModifiedTimeAfter = std::forward<LastModifiedTimeAfterT>(value);
  }
  template <typename LastModifiedTimeAfterT = Aws::Utils::DateTime>
  ListInferenceExperimentsRequest& WithLastModifiedTimeAfter(LastModifiedTimeAfterT&& value) {
    SetLastModifiedTimeAfter(std::forward<LastModifiedTimeAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Selects inference experiments which were last modified before this
   * timestamp.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTimeBefore() const { return m_lastModifiedTimeBefore; }
  inline bool LastModifiedTimeBeforeHasBeenSet() const { return m_lastModifiedTimeBeforeHasBeenSet; }
  template <typename LastModifiedTimeBeforeT = Aws::Utils::DateTime>
  void SetLastModifiedTimeBefore(LastModifiedTimeBeforeT&& value) {
    m_lastModifiedTimeBeforeHasBeenSet = true;
    m_lastModifiedTimeBefore = std::forward<LastModifiedTimeBeforeT>(value);
  }
  template <typename LastModifiedTimeBeforeT = Aws::Utils::DateTime>
  ListInferenceExperimentsRequest& WithLastModifiedTimeBefore(LastModifiedTimeBeforeT&& value) {
    SetLastModifiedTimeBefore(std::forward<LastModifiedTimeBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The column by which to sort the listed inference experiments.</p>
   */
  inline SortInferenceExperimentsBy GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  inline void SetSortBy(SortInferenceExperimentsBy value) {
    m_sortByHasBeenSet = true;
    m_sortBy = value;
  }
  inline ListInferenceExperimentsRequest& WithSortBy(SortInferenceExperimentsBy value) {
    SetSortBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The direction of sorting (ascending or descending).</p>
   */
  inline SortOrder GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrder value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListInferenceExperimentsRequest& WithSortOrder(SortOrder value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The response from the last list when returning a list large enough to need
   * tokening. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListInferenceExperimentsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to select.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListInferenceExperimentsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_nameContains;

  InferenceExperimentType m_type{InferenceExperimentType::NOT_SET};

  InferenceExperimentStatus m_statusEquals{InferenceExperimentStatus::NOT_SET};

  Aws::Utils::DateTime m_creationTimeAfter{};

  Aws::Utils::DateTime m_creationTimeBefore{};

  Aws::Utils::DateTime m_lastModifiedTimeAfter{};

  Aws::Utils::DateTime m_lastModifiedTimeBefore{};

  SortInferenceExperimentsBy m_sortBy{SortInferenceExperimentsBy::NOT_SET};

  SortOrder m_sortOrder{SortOrder::NOT_SET};

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_nameContainsHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_statusEqualsHasBeenSet = false;
  bool m_creationTimeAfterHasBeenSet = false;
  bool m_creationTimeBeforeHasBeenSet = false;
  bool m_lastModifiedTimeAfterHasBeenSet = false;
  bool m_lastModifiedTimeBeforeHasBeenSet = false;
  bool m_sortByHasBeenSet = false;
  bool m_sortOrderHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
