﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/globalaccelerator/GlobalAccelerator_EXPORTS.h>
#include <aws/globalaccelerator/model/Resource.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GlobalAccelerator {
namespace Model {

/**
 * <p>A cross-account attachment in Global Accelerator. A cross-account attachment
 * specifies the <i>principals</i> who have permission to work with
 * <i>resources</i> in your account, which you also list in the
 * attachment.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/Attachment">AWS
 * API Reference</a></p>
 */
class Attachment {
 public:
  AWS_GLOBALACCELERATOR_API Attachment() = default;
  AWS_GLOBALACCELERATOR_API Attachment(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLOBALACCELERATOR_API Attachment& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLOBALACCELERATOR_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the cross-account attachment.</p>
   */
  inline const Aws::String& GetAttachmentArn() const { return m_attachmentArn; }
  inline bool AttachmentArnHasBeenSet() const { return m_attachmentArnHasBeenSet; }
  template <typename AttachmentArnT = Aws::String>
  void SetAttachmentArn(AttachmentArnT&& value) {
    m_attachmentArnHasBeenSet = true;
    m_attachmentArn = std::forward<AttachmentArnT>(value);
  }
  template <typename AttachmentArnT = Aws::String>
  Attachment& WithAttachmentArn(AttachmentArnT&& value) {
    SetAttachmentArn(std::forward<AttachmentArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the cross-account attachment.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Attachment& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The principals included in the cross-account attachment.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPrincipals() const { return m_principals; }
  inline bool PrincipalsHasBeenSet() const { return m_principalsHasBeenSet; }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  void SetPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals = std::forward<PrincipalsT>(value);
  }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  Attachment& WithPrincipals(PrincipalsT&& value) {
    SetPrincipals(std::forward<PrincipalsT>(value));
    return *this;
  }
  template <typename PrincipalsT = Aws::String>
  Attachment& AddPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals.emplace_back(std::forward<PrincipalsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resources included in the cross-account attachment.</p>
   */
  inline const Aws::Vector<Resource>& GetResources() const { return m_resources; }
  inline bool ResourcesHasBeenSet() const { return m_resourcesHasBeenSet; }
  template <typename ResourcesT = Aws::Vector<Resource>>
  void SetResources(ResourcesT&& value) {
    m_resourcesHasBeenSet = true;
    m_resources = std::forward<ResourcesT>(value);
  }
  template <typename ResourcesT = Aws::Vector<Resource>>
  Attachment& WithResources(ResourcesT&& value) {
    SetResources(std::forward<ResourcesT>(value));
    return *this;
  }
  template <typename ResourcesT = Resource>
  Attachment& AddResources(ResourcesT&& value) {
    m_resourcesHasBeenSet = true;
    m_resources.emplace_back(std::forward<ResourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the cross-account attachment was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  Attachment& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the cross-account attachment was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTime() const { return m_createdTime; }
  inline bool CreatedTimeHasBeenSet() const { return m_createdTimeHasBeenSet; }
  template <typename CreatedTimeT = Aws::Utils::DateTime>
  void SetCreatedTime(CreatedTimeT&& value) {
    m_createdTimeHasBeenSet = true;
    m_createdTime = std::forward<CreatedTimeT>(value);
  }
  template <typename CreatedTimeT = Aws::Utils::DateTime>
  Attachment& WithCreatedTime(CreatedTimeT&& value) {
    SetCreatedTime(std::forward<CreatedTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_attachmentArn;
  bool m_attachmentArnHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Vector<Aws::String> m_principals;
  bool m_principalsHasBeenSet = false;

  Aws::Vector<Resource> m_resources;
  bool m_resourcesHasBeenSet = false;

  Aws::Utils::DateTime m_lastModifiedTime{};
  bool m_lastModifiedTimeHasBeenSet = false;

  Aws::Utils::DateTime m_createdTime{};
  bool m_createdTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlobalAccelerator
}  // namespace Aws
