# frozen_string_literal: true

require_relative 'table'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      module ClickHouse
        class CategoryOwnersTable < GitlabQuality::TestTooling::CodeCoverage::ClickHouse::Table
          TABLE_NAME = "category_owners"

          MissingMappingError = Class.new(StandardError)

          def truncate
            logger.debug("#{LOG_PREFIX} Truncating table #{full_table_name} ...")

            client.query("TRUNCATE TABLE #{full_table_name}")

            logger.info("#{LOG_PREFIX} Successfully truncated table #{full_table_name}")
          end

          # Owners of particular feature category as group, stage and section
          #
          # @param feature_category_name [String] the feature_category name
          # @return [Hash]
          def owners(feature_category_name)
            records.fetch(feature_category_name)
          rescue KeyError
            raise(MissingMappingError, "Feature category '#{feature_category_name}' not found in table '#{table_name}'")
          end

          private

          def records
            @records ||= client
              .query("SELECT category, group, stage, section FROM #{table_name}")
              .each_with_object({}) { |record, hsh| hsh[record["category"]] = record.slice("group", "stage", "section") }
          end

          # @return [Boolean] True if the record is valid, false otherwise
          def valid_record?(record)
            required_fields = %i[category group stage section]

            required_fields.each do |field|
              if record[field].nil?
                logger.warn("#{LOG_PREFIX} Skipping record with nil #{field}: #{record}")
                return false
              end
            end

            true
          end
        end
      end
    end
  end
end
