﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/forecast/ForecastService_EXPORTS.h>
#include <aws/forecast/model/ExplainabilityConfig.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ForecastService {
namespace Model {

/**
 * <p>Provides a summary of the Explainability properties used in the
 * <a>ListExplainabilities</a> operation. To get a complete set of properties, call
 * the <a>DescribeExplainability</a> operation, and provide the listed
 * <code>ExplainabilityArn</code>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ExplainabilitySummary">AWS
 * API Reference</a></p>
 */
class ExplainabilitySummary {
 public:
  AWS_FORECASTSERVICE_API ExplainabilitySummary() = default;
  AWS_FORECASTSERVICE_API ExplainabilitySummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_FORECASTSERVICE_API ExplainabilitySummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FORECASTSERVICE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Explainability.</p>
   */
  inline const Aws::String& GetExplainabilityArn() const { return m_explainabilityArn; }
  inline bool ExplainabilityArnHasBeenSet() const { return m_explainabilityArnHasBeenSet; }
  template <typename ExplainabilityArnT = Aws::String>
  void SetExplainabilityArn(ExplainabilityArnT&& value) {
    m_explainabilityArnHasBeenSet = true;
    m_explainabilityArn = std::forward<ExplainabilityArnT>(value);
  }
  template <typename ExplainabilityArnT = Aws::String>
  ExplainabilitySummary& WithExplainabilityArn(ExplainabilityArnT&& value) {
    SetExplainabilityArn(std::forward<ExplainabilityArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the Explainability.</p>
   */
  inline const Aws::String& GetExplainabilityName() const { return m_explainabilityName; }
  inline bool ExplainabilityNameHasBeenSet() const { return m_explainabilityNameHasBeenSet; }
  template <typename ExplainabilityNameT = Aws::String>
  void SetExplainabilityName(ExplainabilityNameT&& value) {
    m_explainabilityNameHasBeenSet = true;
    m_explainabilityName = std::forward<ExplainabilityNameT>(value);
  }
  template <typename ExplainabilityNameT = Aws::String>
  ExplainabilitySummary& WithExplainabilityName(ExplainabilityNameT&& value) {
    SetExplainabilityName(std::forward<ExplainabilityNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Predictor or Forecast used to create
   * the Explainability.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  ExplainabilitySummary& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration settings that define the granularity of time series and
   * time points for the Explainability.</p>
   */
  inline const ExplainabilityConfig& GetExplainabilityConfig() const { return m_explainabilityConfig; }
  inline bool ExplainabilityConfigHasBeenSet() const { return m_explainabilityConfigHasBeenSet; }
  template <typename ExplainabilityConfigT = ExplainabilityConfig>
  void SetExplainabilityConfig(ExplainabilityConfigT&& value) {
    m_explainabilityConfigHasBeenSet = true;
    m_explainabilityConfig = std::forward<ExplainabilityConfigT>(value);
  }
  template <typename ExplainabilityConfigT = ExplainabilityConfig>
  ExplainabilitySummary& WithExplainabilityConfig(ExplainabilityConfigT&& value) {
    SetExplainabilityConfig(std::forward<ExplainabilityConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the Explainability. States include: </p> <ul> <li> <p>
   * <code>ACTIVE</code> </p> </li> <li> <p> <code>CREATE_PENDING</code>,
   * <code>CREATE_IN_PROGRESS</code>, <code>CREATE_FAILED</code> </p> </li> <li> <p>
   * <code>CREATE_STOPPING</code>, <code>CREATE_STOPPED</code> </p> </li> <li> <p>
   * <code>DELETE_PENDING</code>, <code>DELETE_IN_PROGRESS</code>,
   * <code>DELETE_FAILED</code> </p> </li> </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  ExplainabilitySummary& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about any errors that may have occurred during the Explainability
   * creation process.</p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  ExplainabilitySummary& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the Explainability was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  ExplainabilitySummary& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time the resource was modified. The timestamp depends on the status
   * of the job:</p> <ul> <li> <p> <code>CREATE_PENDING</code> - The
   * <code>CreationTime</code>.</p> </li> <li> <p> <code>CREATE_IN_PROGRESS</code> -
   * The current timestamp.</p> </li> <li> <p> <code>CREATE_STOPPING</code> - The
   * current timestamp.</p> </li> <li> <p> <code>CREATE_STOPPED</code> - When the job
   * stopped.</p> </li> <li> <p> <code>ACTIVE</code> or <code>CREATE_FAILED</code> -
   * When the job finished or failed.</p> </li> </ul>
   */
  inline const Aws::Utils::DateTime& GetLastModificationTime() const { return m_lastModificationTime; }
  inline bool LastModificationTimeHasBeenSet() const { return m_lastModificationTimeHasBeenSet; }
  template <typename LastModificationTimeT = Aws::Utils::DateTime>
  void SetLastModificationTime(LastModificationTimeT&& value) {
    m_lastModificationTimeHasBeenSet = true;
    m_lastModificationTime = std::forward<LastModificationTimeT>(value);
  }
  template <typename LastModificationTimeT = Aws::Utils::DateTime>
  ExplainabilitySummary& WithLastModificationTime(LastModificationTimeT&& value) {
    SetLastModificationTime(std::forward<LastModificationTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_explainabilityArn;

  Aws::String m_explainabilityName;

  Aws::String m_resourceArn;

  ExplainabilityConfig m_explainabilityConfig;

  Aws::String m_status;

  Aws::String m_message;

  Aws::Utils::DateTime m_creationTime{};

  Aws::Utils::DateTime m_lastModificationTime{};
  bool m_explainabilityArnHasBeenSet = false;
  bool m_explainabilityNameHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
  bool m_explainabilityConfigHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_messageHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_lastModificationTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ForecastService
}  // namespace Aws
