/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2002 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome Team Members:
 *   Jean Van Wyk <jeanvanwyk@iname.com>
 *   Jeffrey S. Morgan <jeffrey.morgan@bristolwest.com>
 *   Dan Bornstein <danfuzz@milk.com>
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk.event;

import org.gnu.gdk.EventExpose;
import org.gnu.gdk.EventNoExpose;
import org.gnu.gdk.Rectangle;
import org.gnu.gdk.Region;
import org.gnu.gdk.Window;
import org.gnu.glib.EventType;

/**
 * This event object is used to identify how expose has changed.
 * 
 * @author Jonas Berlin
 * @see ExposeListener
 */
public class ExposeEvent extends GtkEvent {

    private final Window window;

    private final boolean sendEvent;

    private final Rectangle area;

    private final Region region;

    private final int count;

    public static class Type extends EventType {
        private Type(int id, String name) {
            super(id, name);
        }

        /**
         * Generated when all or part of a window becomes visible and needs to
         * be redrawn.
         */
        public static final Type EXPOSE = new Type(1, "EXPOSE");

        /**
         * Generated when the area of a Drawable being copied, with
         * Drawable.drawDrawable() or Window.copyArea(), was completely
         * available.
         */
        public static final Type NO_EXPOSE = new Type(2, "NO_EXPOSE");
    }

    public ExposeEvent(Object source, EventExpose gdkEvent) {
        super(source, Type.EXPOSE);
        window = gdkEvent.getWindow();
        sendEvent = gdkEvent.getSendEvent();
        area = gdkEvent.getArea();
        region = gdkEvent.getRegion();
        count = gdkEvent.getCount();
    }

    public ExposeEvent(Object source, EventNoExpose gdkEvent) {
        super(source, Type.NO_EXPOSE);
        window = gdkEvent.getWindow();
        sendEvent = gdkEvent.getSendEvent();
        area = null;
        region = null;
        count = -1;
    }

    /**
     * The window which received the event.
     * 
     * @return the window which received the event.
     */
    public Window getWindow() {
        return window;
    }

    /**
     * Check if the event was sent explicitly (eg using XSendEvent).
     * 
     * @return true if the event was sent explicitly (e.g. using XSendEvent),
     *         false otherwise.
     */
    public boolean getSendEvent() {
        return sendEvent;
    }

    /**
     * Returns the bounding box of region.
     * 
     * @return the bounding box of region.
     */
    public Rectangle getArea() {
        return area;
    }

    /**
     * Returns the region that needs to be redrawn.
     * 
     * @return the region that needs to be redrawn.
     */
    public Region getRegion() {
        return region;
    }

    /**
     * Returns the number of contiguous EXPOSE events following this one. The
     * only use for this is "exposure compression", i.e. handling all contiguous
     * EXPOSE events in one go, though GDK performs some exposure compression so
     * this is not normally needed.
     * 
     * @return the number of contiguous EXPOSE events following this one.
     */
    public int getCount() {
        return count;
    }

    /**
     * Test to compare events.
     */
    public boolean isOfType(ExposeEvent.Type test) {
        return type.getID() == test.getID();
    }
}
