/*
 * Copyright (C) 2004 Imendio HB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <glib/gi18n.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>
#include <gconf/gconf-client.h>
#include <gst/gst.h>
#include <gst/control/control.h>
#include "main-window.h"
#include "source-database.h"
#include "stock-icons.h"
#include "utils.h"
#include "string-utils.h"
#include "player.h"
#include "bacon-message-connection.h"

#ifdef HAVE_DBUS
#include "dbus.h"
#endif

GConfClient *gconf_client;

static void
ensure_dir (void)
{
	char *dir;
	
	dir = g_build_filename (g_get_home_dir (), ".gnome2", NULL);
	
	if (!g_file_test (dir, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)) {
		mkdir (dir, 0755);
	}
	
	g_free (dir);
	
	dir = g_build_filename (g_get_home_dir (), ".gnome2", "jamboree", NULL);
	
	if (!g_file_test (dir, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)) {
		mkdir (dir, 0755);
	}
	
	g_free (dir);
}

static void
bacon_func (const char *message,
	    gpointer    user_data)
{
	MainWindow *window;

	window = user_data;

	if (strcmp (message, "toggle-play") == 0) {
		main_window_handle_play (window);
	}
	else if (strcmp (message, "stop") == 0) {
		main_window_handle_stop (window);
		player_stop ();
	}
	else if (strcmp (message, "prev") == 0) {
		main_window_handle_prev (window);
	}
	else if (strcmp (message, "next") == 0) {
		main_window_handle_next (window);
	}
	else if (strcmp (message, "toggle-hidden") == 0) {
		main_window_toggle_visibility (window);
	}
	else if (strcmp (message, "select-playing-artist") == 0) {
		main_window_select_playing_artist (window);
	}
	else if (strcmp (message, "show") == 0) {
		gtk_window_present (GTK_WINDOW (window));
	}
	else if (strcmp (message, "quit") == 0) {
		gtk_main_quit ();
	} else {
		g_warning ("Unhandled cmd: %s\n", message);
	}
}

int
main (int argc, char **argv)
{
	BaconMessageConnection *conn = NULL;
	int                     arg_version = FALSE;
	char                   *arg_db = NULL;
	int                     arg_toggle_play = FALSE;
	int                     arg_stop = FALSE;
	int                     arg_prev = FALSE;
	int                     arg_next = FALSE;
	int                     arg_hidden = FALSE;
	int                     arg_toggle_hidden = FALSE;
	int                     arg_select_playing_artist = FALSE;
	int                     arg_quit = FALSE;
	gboolean                is_server;
	GnomeProgram           *program;
	Source                 *database;
	GtkWidget              *window;
	char                   *filename;
	
	struct poptOption options[] = {
		{
			"version",
			'v',
			POPT_ARG_NONE,
			&arg_version,
			0,
			N_("Print version"),
			NULL
		},
		{
			"db",
			'\0',
			POPT_ARG_STRING,
			&arg_db,
			0,
			N_("Use an alternative song database file"),
			NULL
		},
		{ 
			"toggle-play",
			't',
			POPT_ARG_NONE,
			&arg_toggle_play,
			0,
			N_("Toggle play and pause mode"),
			NULL
		},
		{ 
			"stop",
			's',
			POPT_ARG_NONE,
			&arg_stop,
			0,
			N_("Stop playing"),
			NULL
		},
		{ 
			"next",
			'n',
			POPT_ARG_NONE,
			&arg_next,
			0,
			N_("Jump to the next song"),
			NULL
		},
		{ 
			"previous",
			'p',
			POPT_ARG_NONE,
			&arg_prev,
			0,
			N_("Jump to the previous song"),
			NULL
		},
		{ 
			"hidden",
			'h',
			POPT_ARG_NONE,
			&arg_hidden,
			0,
			N_("Start up without a visible window"),
			NULL
		},
		{ 
			"toggle-hidden",
			'\0',
			POPT_ARG_NONE,
			&arg_toggle_hidden,
			0,
			N_("Toggle visibility of the window"),
			NULL
		},
		{ 
			"select-playing-artist",
			'\0',
			POPT_ARG_NONE,
			&arg_select_playing_artist,
			0,
			N_("Select the currently playing artist"),
			NULL
		},
		{ 
			"quit",
			'q',
			POPT_ARG_NONE,
			&arg_quit,
			0,
			N_("Quit any running instance"),
			NULL
		},
		POPT_TABLEEND
	};
 	
	bindtextdomain (GETTEXT_PACKAGE, JAMBOREE_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	program = gnome_program_init (PACKAGE, VERSION,
				      LIBGNOMEUI_MODULE, argc, argv,
				      GNOME_PARAM_HUMAN_READABLE_NAME, "Jamboree",
				      GNOME_PARAM_APP_DATADIR, DATADIR,
				      GNOME_PARAM_POPT_TABLE, options,
				      /* Try workaround broken libgnomeui. If
				       * esd is used for events while we're
				       * using esdsink we get nice deadlocks.
				       */
				      GNOME_PARAM_ENABLE_SOUND, FALSE,
				      NULL);
	
	if (arg_version) {
		g_print ("Jamboree " VERSION "\n");
		exit (1);
	}

	/* If the database != the regular one, don't setup a server or try to
	 * contact one.
	 */
	if (!arg_db) {
 		conn = bacon_message_connection_new ("jamboree");
		is_server = bacon_message_connection_get_is_server (conn);

		/* First, handle the client case (toggle, stop, next, prev). */
		if (!is_server) {
			if (arg_toggle_play) {
				bacon_message_connection_send (conn, "toggle-play");
			}
			else if (arg_stop) {
				bacon_message_connection_send (conn, "stop");
			}
			else if (arg_prev) {
				bacon_message_connection_send (conn, "prev");
			}
			else if (arg_next) {
				bacon_message_connection_send (conn, "next");
			}
			else if (arg_toggle_hidden) {
				bacon_message_connection_send (conn, "toggle-hidden");
			}
			else if (arg_select_playing_artist) {
				bacon_message_connection_send (conn, "select-playing-artist");
			}
			else if (arg_quit) {
				bacon_message_connection_send (conn, "quit");
			}
			else if (arg_hidden) {
				/* This does not make sense in client mode. */
			} else {
				bacon_message_connection_send (conn, "show");
			}
			
			gdk_notify_startup_complete ();
			
			return 0;
		} else {
			/* These do not make sense in server mode. */
			if (arg_toggle_hidden || arg_quit || arg_prev || arg_next || arg_stop ||
			    arg_select_playing_artist) {
				gdk_notify_startup_complete ();
				return 0;
			}
		}
	}
		
	gst_init (&argc, &argv);
	gst_control_init (&argc, &argv);

	shared_string_init ();

	g_random_set_seed (time (NULL));

	ensure_dir ();
	
	if (arg_db != NULL) {
		filename = g_strdup (arg_db);
	} else {
		filename = g_build_filename (g_get_home_dir (),
					     ".gnome2", "jamboree", "songs2.db",
					     NULL);
	}

	database = source_database_new (filename);
	if (!database) {
		GtkWidget *dialog;
		
		gdk_notify_startup_complete ();
		
		dialog = hig_dialog_new (NULL,
					 0,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_OK,
					 _("Could not read song database"),
					 _("Jamboree is most likely already running."));
		
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		
		exit (1);
	}
  
	g_free (filename);

#ifdef HAVE_DBUS
	jamboree_dbus_init_service ();
#endif
  
	gconf_client = gconf_client_get_default ();
	gconf_client_add_dir (gconf_client, GCONF_PATH,
			      GCONF_CLIENT_PRELOAD_ONELEVEL,
			      NULL);
	
	stock_icons_register ();
	window = main_window_new (database);

	g_object_unref (database);

	if (conn) {
		bacon_message_connection_set_callback (conn, bacon_func, window);
	}
	
	if (!arg_hidden) {
		gtk_widget_show (window);
	}

	g_signal_connect (window,
			  "destroy", G_CALLBACK (gtk_main_quit),
			  NULL);
	
	gtk_main ();

	if (conn) {
		bacon_message_connection_free (conn);
	}
	
	shared_string_shutdown ();

	g_object_unref (gconf_client);

	return 0;
}
