/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scanner
 * @short_description: Acquisition plugin
 *
 * #GnomeScanner is the base class for backend specific scanner (like
 * e.g SANE). Following SANE design, Gnome Scan scanner has an
 * undefined set of option. But unlike SANE, the scanner itself handle
 * acquisition.
 *
 * Each #GnomeScanner should monitor its sensor (e.g. paper-in) and
 * trigger an event on sensor state changes.
 **/

#include <glib/gi18n.h>
#include "gnome-scanner.h"
#include "gnome-scan-types.h"
#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCANNER, GnomeScannerPrivate))

typedef struct _GnomeScannerPrivate GnomeScannerPrivate;

struct _GnomeScannerPrivate
{
  const gchar*	icon_name;
};

enum
  {
    PROP_0,
    PROP_SENSORS,
    PROP_ICON_NAME,
    PROP_STATUS
  };

enum
  {
    EVENT,
    SETTINGS_CHANGED,
    STATUS_CHANGED,
    LAST_SIGNAL
  };


static GnomeScanPluginClass* parent_class = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (GnomeScanner, gnome_scanner, GNOME_TYPE_SCAN_PLUGIN);

static void
gnome_scanner_init (GnomeScanner *object)
{
}

static void
gnome_scanner_finalize (GObject *object)
{
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scanner_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (GNOME_IS_SCANNER (object));

  switch (prop_id)
    {
    case PROP_SENSORS:
      /* TODO: Add setter for "sensors" property here */
      break;
    case PROP_ICON_NAME:
      GET_PRIVATE (object)->icon_name = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gnome_scanner_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (GNOME_IS_SCANNER (object));

  switch (prop_id)
    {
    case PROP_SENSORS:
      /* TODO: Add getter for "sensors" property here */
      break;
    case PROP_ICON_NAME:
      g_value_set_string (value, gnome_scanner_get_icon_name (GNOME_SCANNER (object)));
      break;
    case PROP_STATUS:
      g_value_set_enum (value, GNOME_SCANNER (object)->status);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gnome_scanner_event (GnomeScanner *scanner, GnomeScanEvent *event, gpointer data)
{
  /* TODO: Add default signal handler implementation here */
}

static void
gnome_scanner_class_init (GnomeScannerClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  parent_class = GNOME_SCAN_PLUGIN_CLASS (g_type_class_peek_parent (klass));

  g_type_class_add_private (klass, sizeof (GnomeScannerPrivate));
  object_class->finalize = gnome_scanner_finalize;
  object_class->set_property = gnome_scanner_set_property;
  object_class->get_property = gnome_scanner_get_property;

  klass->event = gnome_scanner_event;

  /**
   * GnomeScanner:sensors:
   *
   * A list of sensor specification.
   **/
  g_object_class_install_property (object_class,
				   PROP_SENSORS,
				   g_param_spec_pointer ("sensors",
							 "Sensors",
							 "Device sensors specification.",
							 G_PARAM_READABLE));
	
  /**
   * GnomeScanner:icon-name:
   *
   * The icon name representing the device.
   **/
  g_object_class_install_property (object_class,
				   PROP_ICON_NAME,
				   g_param_spec_string ("icon-name",
							"Icon Name",
							"Icon name representing device type.",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
  /**
   * GnomeScanner:status:
   *
   * Device status
   **/
  g_object_class_install_property (object_class,
				   PROP_STATUS,
				   g_param_spec_enum ("status",
						      "Status",
						      "Current device status",
						      GNOME_TYPE_SCANNER_STATUS,
						      GNOME_SCANNER_UNKNOWN,
						      G_PARAM_READABLE));

  /**
   * GnomeScanner::event:
   * @scanner:	the emitting #GnomeScanner
   * @event:	the emitted even.
   *
   * Emitted when an hardware event is triggered. (button, wheel, paper-in, etc.)
   **/
  signals[EVENT] =
    g_signal_new ("event",
		  G_OBJECT_CLASS_TYPE (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GnomeScannerClass, event),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);
	
  /**
   * GnomeScanner::settings-changed:
   * @scanner:	the emitting #GnomeScanner
   *
   * Emitted when the scanner has settings are hard setted.
   **/
  signals[SETTINGS_CHANGED] =
    g_signal_new ("settings-changed",
		  G_OBJECT_CLASS_TYPE (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GnomeScannerClass, settings_changed),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  /**
   * GnomeScanner::status-changed:
   * @scanner:	the emitting #GnomeScanner
   *
   * Emitted when the scanner status as changed.
   **/
  signals[STATUS_CHANGED] =
    g_signal_new ("status-changed",
		  G_OBJECT_CLASS_TYPE (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GnomeScannerClass, status_changed),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
}


/**
 * gnome_scanner_get_icon_name:
 * @scanner: a #GnomeScanner
 * 
 * Return the icon-name representing the type of the device.
 * 
 * Returns: a icon-name string
 **/
const gchar*
gnome_scanner_get_icon_name (GnomeScanner *scanner)
{
  return GET_PRIVATE (scanner)->icon_name;
}

/**
 * gnome_scanner_settings_changed:
 * @scanner: a #GnomeScanner
 *
 * Emit #GnomeScanner::settings-changed signal.
 *
 */
void
gnome_scanner_settings_changed (GnomeScanner *scanner)
{
  g_signal_emit (scanner, signals[SETTINGS_CHANGED], 0);
}


GnomeScannerStatus
gnome_scanner_get_status			(GnomeScanner *scanner)
{
  return GNOME_SCANNER (scanner)->status;
}
																
/**
 * gnome_scanner_get_status_string:
 * @scanner: a #GnomeScanner
 *
 * Returns the translated string for the current @scanner status.
 *
 * Returns: a translated string.
 **/
const gchar*
gnome_scanner_get_status_string		(GnomeScanner *scanner)
{
  static const gchar* status_strings[] = {
    N_("Unknown"),
    N_("Ready"),
    N_("Busy"),
    NULL
  };
  return _(status_strings[scanner->status]);
}


void
gnome_scanner_set_status (GnomeScanner *scanner, GnomeScannerStatus status)
{
  if (scanner->status != status) {
    scanner->status = status;
    g_signal_emit (scanner, signals[STATUS_CHANGED], 0);
  }
}

const gchar*
gnome_scanner_get_output_format	(GnomeScanner *scanner)
{
  GnomeScannerClass *klass = GNOME_SCANNER_GET_CLASS (scanner);
  if (klass->get_output_format) {
    return klass->get_output_format(scanner);
  }
  else {
    return NULL;
  }
}
