/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-plugin
 * @short_description: Scan operation plugin
 *
 * A #GnomeScanPlugin add a features to Gnome Scan. A plugin never
 * extends directly #GnomeScanPlugin. Instead, you should see
 * #GnomeScanner and #GnomeScanSink. Basically, a plugin handle common
 * properties such as @name, @blurb and @params. Each plugins as a per
 * instance list of params.
 *
 * In the future, #GnomeScanPlugin should be drop in favor of
 * #GeglOperation.
 **/

#include "gnome-scan-plugin.h"
#include "gnome-scan-param-specs.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_PLUGIN, GnomeScanPluginPrivate))

typedef struct _GnomeScanPluginPrivate GnomeScanPluginPrivate;

struct _GnomeScanPluginPrivate
{
  gchar*			name;
  gchar*			blurb;
};

enum
  {
    PROP_0,
    PROP_NAME,
    PROP_BLURB,
    PROP_PARAMS
  };

enum
  {
    PARAMS_CHANGED,
    LAST_SIGNAL
  };

static GObjectClass* parent_class = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (GnomeScanPlugin, gnome_scan_plugin, G_TYPE_OBJECT);

static void
gnome_scan_plugin_init (GnomeScanPlugin *object)
{
  object->params = g_param_spec_pool_new (FALSE);
}

static void
gnome_scan_plugin_finalize (GObject *object)
{
  /* TODO: Add deinitalization code here */

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_plugin_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (object));

  switch (prop_id)
    {
    case PROP_NAME:
      GET_PRIVATE (object)->name = g_value_dup_string (value);
      break;
    case PROP_BLURB:
      GET_PRIVATE (object)->blurb = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gnome_scan_plugin_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (object));

  switch (prop_id)
    {
    case PROP_NAME:
      g_value_set_string (value, gnome_scan_plugin_get_name (GNOME_SCAN_PLUGIN (object)));
      break;
    case PROP_BLURB:
      g_value_set_string (value, GET_PRIVATE (object)->blurb);
      break;
    case PROP_PARAMS:
      g_value_set_pointer (value, GNOME_SCAN_PLUGIN (object)->params);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gnome_scan_plugin_class_init (GnomeScanPluginClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

  g_type_class_add_private (klass, sizeof (GnomeScanPluginPrivate));
  object_class->finalize = gnome_scan_plugin_finalize;
  object_class->set_property = gnome_scan_plugin_set_property;
  object_class->get_property = gnome_scan_plugin_get_property;

  /**
   * GnomeScanPlugin:name:
   *
   * The plugin public name.
   **/
  g_object_class_install_property (object_class,
				   PROP_NAME,
				   g_param_spec_string ("name",
							"Name",
							"Plugin public name",
							NULL,
							G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  /**
   * GnomeScanPlugin:blurb:
   *
   * The plugin public blurb.
   **/
  g_object_class_install_property (object_class,
				   PROP_BLURB,
				   g_param_spec_string ("blurb",
							"Blurb",
							"Plugin public description.",
							NULL,
							G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  /**
   * GnomeScanPlugin:params:
   *
   * The list of per instance #GSParamSpec.
   **/
  g_object_class_install_property (object_class,
				   PROP_PARAMS,
				   g_param_spec_pointer ("params",
							 "Parameters",
							 "Pool of plugin parameters specification.",
							 G_PARAM_READABLE));
	
  /**
   * GnomeScanPlugin::changed:
   * @plugin: the emitting #GnomeScanPlugin
   *
   * Triggered when the instance parameters list has changed.
   **/
  signals[PARAMS_CHANGED] =
    g_signal_new ("params-changed",
		  G_OBJECT_CLASS_TYPE (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GnomeScanPluginClass, changed),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);
}


/**
 * gnome_scan_plugin_get_name:
 * @plugin: a #GnomeScanPlugin
 * 
 * Returns: the @plugin name
 **/
gchar*
gnome_scan_plugin_get_name (GnomeScanPlugin *plugin)
{
  g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
  return g_strdup (GET_PRIVATE (plugin)->name);
}

/**
 * gnome_scan_plugin_params_add:
 * @plugin: a #GnomeScanPlugin
 * @param: a #GSParamSpec
 * 
 * See: %gnome-scan-param-specs
 **/
void
gnome_scan_plugin_params_add (GnomeScanPlugin *plugin, GParamSpec *param)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
  g_param_spec_pool_insert (plugin->params, param, G_OBJECT_TYPE (plugin));
  gnome_scan_plugin_params_changed (plugin, param);
}

/**
 * gnome_scan_plugin_params_remove:
 * @plugin: a #GnomeScanPlugin
 * @param: a #GSParamSpec
 * 
 * Remove a param spec from the param list.
 **/
void
gnome_scan_plugin_params_remove (GnomeScanPlugin *plugin, GParamSpec *param)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
  g_param_spec_pool_remove (plugin->params, param);
  gnome_scan_plugin_params_changed (plugin, NULL);
}

/**
 * gnome_scan_plugin_params_clear:
 * @plugin: a #GnomeScanPlugin
 * 
 * Remove all instance parameters from a plugin.
 **/
void
gnome_scan_plugin_params_clear (GnomeScanPlugin *plugin)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
  GList *node;
	
  node = g_param_spec_pool_list_owned (plugin->params, G_OBJECT_TYPE (plugin));

  for (; node; node = node->next)
    g_param_spec_pool_remove (plugin->params, node->data);
	
  gnome_scan_plugin_params_changed (plugin, NULL);
}

/**
 * gnome_scan_plugin_get_params:
 * @plugin: a #GnomeScanPlugin
 * 
 * Retrieve all instance paramters from @plugin.
 * 
 * Returns: a GList of #GSParamSpec.
 **/
GList*
gnome_scan_plugin_get_params (GnomeScanPlugin *plugin)
{
  g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
  GList *params = NULL;
	
  params = g_param_spec_pool_list_owned (plugin->params,
					 G_OBJECT_TYPE (plugin));
  params = g_list_sort (params, (GCompareFunc) gs_param_spec_cmp_index);
  return params;
}

GSList*
gnome_scan_plugin_params_get_groups (GnomeScanPlugin *plugin)
{
  GSList* groups = NULL;
  GList* node = gnome_scan_plugin_get_params(plugin);
  GQuark cur_group = 0, last_group = 0;
	
  for (; node ; node = node->next) {
    cur_group = gs_param_spec_get_group (node->data);
    if (!last_group || last_group != cur_group) {
      groups = g_slist_append (groups, (gpointer) cur_group);
      last_group = cur_group;
    }
  }
	
  return groups;
}

GSList*
gnome_scan_plugin_params_get_other_groups (GnomeScanPlugin *plugin, GQuark known_group0, ...)
{
  va_list args;
  va_start (args, known_group0);
  GSList *groups = gnome_scan_plugin_params_get_groups (plugin);
  GQuark known_group;
	
  /* list known group not to list */
  known_group = known_group0;
  while (known_group) {
    groups = g_slist_remove_all (groups, (gpointer) known_group);
    known_group = va_arg (args, GQuark);
  }
	
  return groups;
}

/**
 * gnome_scan_plugin_get_param_group:
 * @plugin: a #GnomeScanPlugin
 * @group: a group #GQuark
 * 
 * Return all instance parameters which have @group as group.
 * 
 * Returns: a GList of #GSParamSpec.
 **/
GSList*
gnome_scan_plugin_get_param_group (GnomeScanPlugin *plugin, GQuark group)
{
  g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
  GSList *list = NULL;
  gint i;
  guint n;
  GQuark pgroup;
  GParamSpec **specs =
    g_param_spec_pool_list (plugin->params,
			    G_OBJECT_TYPE (plugin), &n);
	
  for (i = 0; i < n; i++) {
    pgroup = gs_param_spec_get_group (specs[i]);
    if (pgroup == group)
      list = g_slist_append (list, specs[i]);
  }

  list = g_slist_sort (list, (GCompareFunc) gs_param_spec_cmp_index);
	
  return list;
}

/* List all param for groups0 and following (terminated by 0). */
GSList*
gnome_scan_plugin_get_param_groups	(GnomeScanPlugin *plugin, GQuark group0, ...)
{
  GSList *groups = NULL;
  GQuark group;
  va_list args;
  va_start(args, group0);

  group = group0;
  while (group) {
    groups = g_slist_concat (groups,
			     gnome_scan_plugin_get_param_group (plugin, group));
    group = va_arg(args, GQuark);
  }
	
  return groups;
}

GParamSpec*
gnome_scan_plugin_params_lookup	(GnomeScanPlugin *plugin, const gchar *name)
{
  return g_param_spec_pool_lookup (plugin->params,
				   name,
				   G_OBJECT_TYPE (plugin),
				   FALSE);
}

void	gnome_scan_plugin_params_foreach (GnomeScanPlugin *plugin, GFunc func, gpointer user_data)
{
  GList *list = g_param_spec_pool_list_owned (plugin->params, G_OBJECT_TYPE (plugin));
  g_list_foreach (list, func, user_data);
}

#define	gs_exec_class_func_ret(k,f,d,a...)	if(k->f){return klass->f(a);}else{return d;}
#define	gs_exec_class_func(k,f,a...)		if(k->f){klass->f(a);}

/**
 * gnome_scan_plugin_configure:
 * @plugin: a #GnomeScanPlugin
 * @settings: a #GnomeScanPlugin
 * 
 * Search in @settings value for its instance parameters.
 **/
void
gnome_scan_plugin_configure (GnomeScanPlugin *plugin, GnomeScanSettings *settings)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin) && GNOME_IS_SCAN_SETTINGS (settings));
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  gs_exec_class_func(klass, configure, plugin, settings);
}

GList*
gnome_scan_plugin_get_child_nodes (GnomeScanPlugin *plugin,
				   GeglNode *root)
{
  g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  gs_exec_class_func_ret(klass, get_child_nodes, NULL, plugin, root);
}

void
gnome_scan_plugin_configure_frame	(GnomeScanPlugin *plugin)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  gs_exec_class_func(klass, configure_frame, plugin)
}

gboolean
gnome_scan_plugin_start_frame	(GnomeScanPlugin *plugin)
{
  g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), FALSE);
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  gs_exec_class_func_ret(klass, start_frame, TRUE, plugin);
}

/**
 * gnome_scan_plugin_work:
 * @plugin: a #GnomeScanPlugin
 * @progress: a pointer to a place where to store progress
 * 
 * Do an iteration of plugin work. Store the amount of work done in
 * progress as a fraction, from 0 to 1. progress set to 1 means work
 * completed.
 *
 * Returns: Wether more work iteration needs to be done.
 **/
gboolean
gnome_scan_plugin_work (GnomeScanPlugin *plugin, gdouble *progress)
{
  g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), FALSE);
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  *progress = 1.;
  gs_exec_class_func_ret(klass, work, FALSE, plugin, progress);
}


void
gnome_scan_plugin_end_frame		(GnomeScanPlugin *plugin)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  gs_exec_class_func(klass, end_frame, plugin);
}


void
gnome_scan_plugin_end_scan		(GnomeScanPlugin *plugin)
{
  g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
  GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
  gs_exec_class_func(klass, end_scan, plugin);
}

/**
 * gnome_scan_plugin_changed:
 * @plugin: a #GnomeScanPlugin
 *
 * Emit #GnomeScanPlugin::changed
 */
void
gnome_scan_plugin_params_changed	(GnomeScanPlugin *plugin, GParamSpec *spec)
{
  g_signal_emit (plugin, signals[PARAMS_CHANGED], 0, spec);
}

