/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-param-widget
 * @title:	GnomeScanParamWidget
 * @short_description: A generic parameter widget
 *
 * This widget build it self automatically considering the #GParamSpec
 * passed to the constructor.
 **/

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <glib/gi18n.h>
#include "gnome-scan-param-widget.h"
#include "gnome-scan-param-specs.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_PARAM_WIDGET, GnomeScanParamWidgetPrivate))
#define I18N(gspw,text)	dgettext (gs_param_spec_get_domain (GET_PRIVATE ((gspw))->pspec), text)

typedef struct _GnomeScanParamWidgetPrivate GnomeScanParamWidgetPrivate;
struct _GnomeScanParamWidgetPrivate
{
	GtkWidget	*unit;
	gboolean	propagate;
	
	gulong		settings_changed_handler;
};

enum
{
	PROP_0,
	PROP_SETTINGS,
	PROP_PLUGIN,
	PROP_PARAM_SPEC,
	PROP_VALUE
};

static GtkHBoxClass* parent_class = NULL;

void	gspw_settings_changed (GnomeScanSettings *settings, gchar *key, GnomeScanParamWidget *gspw);

G_DEFINE_TYPE (GnomeScanParamWidget, gnome_scan_param_widget, GTK_TYPE_HBOX);

static void
gnome_scan_param_widget_init (GnomeScanParamWidget *object)
{
	object->value		= g_new0 (GValue, 1);
}

GObject*
gnome_scan_param_widget_constructor (GType	type, guint	n_params, GObjectConstructParam	*params)
{
	GValue *value;
	gchar* unit;
	GObject *object =
		G_OBJECT_CLASS (parent_class)->constructor (type, n_params, params);
	
	GnomeScanParamWidget *gspw = GNOME_SCAN_PARAM_WIDGET (object);
	GnomeScanParamWidgetPrivate *priv = GET_PRIVATE (gspw);
	GnomeScanParamWidgetClass *klass = GNOME_SCAN_PARAM_WIDGET_GET_CLASS (gspw);
	
	gtk_box_set_spacing (GTK_BOX (gspw), 6);
	
	/* default value */
	value = gnome_scan_settings_get (gspw->settings,
									 g_param_spec_get_name (gspw->pspec));
	g_value_init (gspw->value, G_PARAM_SPEC_VALUE_TYPE (gspw->pspec));
	g_param_value_set_default (gspw->pspec, gspw->value);

	if (value) {
		g_value_transform (value, gspw->value);
	}
	else {
		gnome_scan_settings_set (gspw->settings,
								 g_param_spec_get_name (gspw->pspec),
								 g_boxed_copy (G_TYPE_VALUE,
											   gspw->value));
	}
	
	klass->build (GNOME_SCAN_PARAM_WIDGET (object));
	if (!gspw->shows_unit) {
		priv->unit = gtk_label_new(NULL);
		switch (gs_param_spec_get_unit (gspw->pspec)) {
			case GS_UNIT_PIXEL:
				unit = _("px");
			break;
			case GS_UNIT_BIT:
				unit = _("bit");
			break;
			case GS_UNIT_MM:
				unit = _("mm");
			break;
			case GS_UNIT_DPI:
				unit = _("dpi");
			break;
			case GS_UNIT_PERCENT:
				unit = _("%");
			break;
			case GS_UNIT_MICROSECOND:
				unit = _("ms");
			break;
			case GS_UNIT_NONE:
				default:
				unit = NULL;
			break;
		};
		
		if (unit) {
			priv->unit = gtk_label_new (unit);
			gtk_box_pack_start (GTK_BOX (object), priv->unit, FALSE, FALSE, 0);
		}
	}
	
	klass->set (gspw, gspw->value);
	/*klass->show (gspw);*/
	
	priv->propagate = TRUE;
	priv->settings_changed_handler =
		g_signal_connect (gspw->settings, "changed",
						  (GCallback) gspw_settings_changed,
						  gspw);
	
	return object;
}

static void
gnome_scan_param_widget_finalize (GObject *object)
{
	GnomeScanParamWidget *gspw = GNOME_SCAN_PARAM_WIDGET(object);
	GnomeScanParamWidgetPrivate *priv = GET_PRIVATE (gspw);
	g_signal_handler_disconnect (gspw->settings,
								 priv->settings_changed_handler);
	g_value_unset (gspw->value);
	g_free (gspw->value);
	g_object_unref(gspw->settings);
	g_object_unref(gspw->plugin);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_param_widget_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_PARAM_WIDGET (object));

	GnomeScanParamWidget *gspw = GNOME_SCAN_PARAM_WIDGET (object);
	switch (prop_id)
	{
		case PROP_SETTINGS:
			gspw->settings = GNOME_SCAN_SETTINGS (g_value_dup_object (value));
		break;
		case PROP_PLUGIN:
			gspw->plugin = GNOME_SCAN_PLUGIN (g_value_dup_object (value));
		break;
		case PROP_PARAM_SPEC:
			gspw->pspec = g_value_get_pointer (value);
		break;
		case PROP_VALUE:
			gnome_scan_param_widget_set_value (GNOME_SCAN_PARAM_WIDGET (object),
											   g_value_get_boxed (value));
		break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
gnome_scan_param_widget_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_PARAM_WIDGET (object));

	switch (prop_id)
	{
		case PROP_PARAM_SPEC:
			/* TODO: Add getter for "param-spec" property here */
			break;
		case PROP_VALUE:
			g_value_set_boxed (value,
							   gnome_scan_param_widget_get_value (GNOME_SCAN_PARAM_WIDGET (object)));
		break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
gnome_scan_param_widget_class_init (GnomeScanParamWidgetClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = GTK_HBOX_CLASS (g_type_class_peek_parent (klass));

	g_type_class_add_private (klass, sizeof (GnomeScanParamWidgetPrivate));
	object_class->constructor	= gnome_scan_param_widget_constructor;
	object_class->finalize		= gnome_scan_param_widget_finalize;
	object_class->set_property	= gnome_scan_param_widget_set_property;
	object_class->get_property	= gnome_scan_param_widget_get_property;

	/**
	 * GnomeScanParamWidget:settings:
	 *
	 **/
	g_object_class_install_property (object_class,
									 PROP_SETTINGS,
									 g_param_spec_object ("settings",
														  "Settings",
														  "The #GnomeScanSettings where to store value",
														  GNOME_TYPE_SCAN_SETTINGS,
														  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	
	/**
	 * GnomeScanParamWidget:plugin:
	 *
	 **/
	g_object_class_install_property (object_class,
									 PROP_PLUGIN,
									 g_param_spec_object ("plugin",
														  "Plugin",
														  "The #GnomeScanPlugin the widget is configuring",
														  GNOME_TYPE_SCAN_PLUGIN,
														  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

	/**
	 * GnomeScanParamWidget:param-spec:
	 *
	 * The param spec the widget is representing. A widget can handle
	 * only one #GParamSpec per instance, the widget being builded
	 * upon construction depending on the #GParamSpec.
	 **/
	g_object_class_install_property (object_class,
	                                 PROP_PARAM_SPEC,
	                                 g_param_spec_pointer ("param-spec",
	                                                       "Param Spec",
	                                                       "The parameter specification the widget is handling",
	                                                       G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

	/**
	 * GnomeScanParamWidget:value:
	 *
	 * The current value associated with the param-spec.
	 **/
	g_object_class_install_property (object_class,
	                                 PROP_VALUE,
	                                 g_param_spec_boxed ("value",
	                                                     "Value",
	                                                     "The parameter value.",
	                                                     G_TYPE_VALUE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE));
}

/**
 * gnome_scan_param_widget_new:
 * @pspec: a #GParamSpec
 * 
 * Convenient constructor that retrieve default value from @pspec and
 * pass it to gnome_scan_param_widget_new_with_value().
 * 
 * See: gnome_scan_param_widget_new_with_value()
 * Returns: The new #GnomeScanParamWidget
 **/
GtkWidget*
gnome_scan_param_widget_new (GnomeScanSettings *settings,
							 GnomeScanPlugin *plugin,
							 GParamSpec *pspec)
{
	GType type = gs_param_spec_get_widget_type (pspec);
	
	if (type == G_TYPE_INVALID) {
		g_warning ("%s: No widget type for param %s (%s)", __FUNCTION__,
				   G_PARAM_SPEC_TYPE_NAME (pspec),
				   G_PARAM_SPEC_TYPE_NAME (pspec));
		return NULL;
	}
	
	GObject *object =
		g_object_new (type,
					  "settings", settings,
					  "plugin", plugin,
					  "param-spec", pspec,
					  NULL);
	
	return GTK_WIDGET (object);
}

/**
 * gnome_scan_param_widget_get_param_spec:
 * @widget: a #GnomeScanParamWidget
 * 
 * Retrieve the spec the widget is representing.
 * 
 * Returns: a pointer to the widget param spec
 **/
GParamSpec*
gnome_scan_param_widget_get_param_spec (GnomeScanParamWidget *widget)
{
	return widget->pspec;
}

/**
 * gnome_scan_param_widget_set_value:
 * @widget: a #GnomeScanParamWidget
 * @value: a #GValue
 * 
 * Set the value of the widget. It's up to you to ensure the value
 * correspond to the param spec.
 **/
void
gnome_scan_param_widget_set_value (GnomeScanParamWidget *widget, GValue *value)
{
	g_value_copy (value, widget->value);
	GNOME_SCAN_PARAM_WIDGET_GET_CLASS (widget)->set (widget, value);
}

/**
 * gnome_scan_param_widget_get_value:
 * @widget: a #GnomeScanParamWidget
 * 
 * Retrieve the current value the user setted.
 * 
 * Returns: a #GValue
 **/
GValue*
gnome_scan_param_widget_get_value (GnomeScanParamWidget *widget)
{
	return widget->value;
}

/**
 * gnome_scan_param_widget_shows_label:
 * @widget: a #GnomeScanParamWidget
 * 
 * A #GnomeScanParamWidget can show itself the param nick or other
 * label. If not, the parent container can show the @pspec nickname in
 * e.g. a #GtkTable. This function allow the parent to know wether the
 * label is already shown or not.
 * 
 * Returns: TRUE if the @widget shows the label.
 **/
gboolean
gnome_scan_param_widget_shows_label (GnomeScanParamWidget *widget)
{
	return widget->shows_label;
}

/**
 * gnome_scan_param_widget_expands:
 * @widget: a #GnomeScanParamWidget
 * 
 * This function helps parent to know how to pack/attach the @widget
 * in a container. If the widget is huge (e.g. it contains a preview),
 * then it make sense to expands this widget. This function returns
 * wether this widget should expands or not.
 * 
 * Returns: TRUE if the @widget should expand.
 **/
gboolean
gnome_scan_param_widget_expands (GnomeScanParamWidget *widget)
{
	return widget->expands;
}


void
gnome_scan_param_widget_changed (GnomeScanParamWidget *gspw)
{
	GnomeScanParamWidgetPrivate *priv = GET_PRIVATE (gspw);
	if (priv->propagate) {
		gnome_scan_settings_set (gspw->settings,
								 g_param_spec_get_name (gspw->pspec),
								 gspw->value);
		gnome_scan_plugin_configure (gspw->plugin, gspw->settings);
	}
}

void
gspw_settings_changed (GnomeScanSettings *settings, gchar *key, GnomeScanParamWidget *gspw)
{
	GnomeScanParamWidgetClass *klass = GNOME_SCAN_PARAM_WIDGET_GET_CLASS (gspw);
	GnomeScanParamWidgetPrivate *priv = GET_PRIVATE (gspw);
	
	if (priv->propagate && g_str_equal (key, g_param_spec_get_name (gspw->pspec))) {
		priv->propagate = FALSE;
		klass->set (gspw, gnome_scan_settings_get (settings, key));
		priv->propagate = TRUE;
	}
}
