/* ACME
 * Copyright (C) 2001, 2002 Bastien Nocera <hadess@hadess.net>
 *
 * acme-properties.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 */

#include <sys/file.h>
#include <sys/stat.h>
#include <gnome.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include "acme.h"
#include "eggcellrendererkeys.h"
#include "config.h"

#define BACKSPACE_KEYCODE 22
#define C_KEYCODE 54

enum {
	PATH_COL,
	DESC_COL,
	KEYVAL_COL,
	NUM_COLS
};

GConfClient *conf_client;

static gboolean
is_running ()
{
	char *path;
	int fd;

	path = g_strdup_printf ("%s/.acme.lock", g_get_home_dir());
	fd = open (path, O_WRONLY, S_IRWXU);

	g_free (path);
	if (fd < 0)
		return FALSE;

	if (lockf (fd, F_TEST, 0L) < 0)
	{
		close (fd);
		return TRUE;
	}

	close (fd);
	return FALSE;
}

static void
response_cb (GtkDialog *dialog, gint response, gpointer data)
{
	gtk_main_quit ();
}

static void
checkbox_toggled_cb (GtkWidget *widget, gpointer data)
{
	gboolean state;

	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gconf_client_set_bool (conf_client,
			"/apps/acme/use_pcm", state, NULL);
}

static void
exec_clicked (GtkWidget *widget, gpointer data)
{
	GtkWidget *hbox = (GtkWidget *)data;

	if (g_spawn_command_line_async ("acme", NULL) == TRUE)
		gtk_widget_set_sensitive (hbox, FALSE);
}

static void
keys_edited_callback (GtkCellRendererText *cell, const char *path_string,
		guint keyval, GdkModifierType mask, guint keycode,
		gpointer data)
{
	GtkTreeModel *model = (GtkTreeModel *)data;
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter iter;
	char *gconf_path;
	int value;

	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, PATH_COL, &gconf_path, -1);

	if (gconf_path == NULL)
		return;

	if (keycode == C_KEYCODE)
	{
		gtk_tree_path_free (path);
		return;
	}

	if (keycode == BACKSPACE_KEYCODE)
		value = -1;
	else
		value = keycode;

	gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			KEYVAL_COL, value,
			-1);
	gconf_client_set_int (conf_client,
			gconf_path, value, NULL);

	gtk_tree_path_free (path);
}

static void
init_content_helper (GtkListStore *store, const char *path,
		const char *desc, int keyval)
{
	GtkTreeIter iter;

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			PATH_COL, path,
			DESC_COL, desc,
			KEYVAL_COL, keyval,
			-1);
}

static void
init_content (GtkTreeView *treeview)
{
	GtkListStore *store;
	int i, keycode;

	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));
	for (i = 0; i < HANDLED_KEYS; i++)
	{
		keycode = gconf_client_get_int (conf_client,
				keys[i].key_config, NULL);
		init_content_helper (store, keys[i].key_config,
				keys[i].description, keycode);
	}
}

static void
init_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	/* Labels */
	renderer = gtk_cell_renderer_text_new ();

	column = gtk_tree_view_column_new_with_attributes (_("Description"),
			renderer,
			"text", DESC_COL,
			NULL);

	gtk_tree_view_append_column (treeview, column);

	/* keys */
	renderer = g_object_new (EGG_TYPE_CELL_RENDERER_KEYS,
			"editable", TRUE,
			NULL);
	g_signal_connect (G_OBJECT (renderer), "keys_edited",
			G_CALLBACK (keys_edited_callback),
			gtk_tree_view_get_model (treeview));

	column = gtk_tree_view_column_new_with_attributes (_("Keycode value"),
			renderer,
			"text", KEYVAL_COL,
			NULL);
	gtk_tree_view_column_set_fixed_width (column, 20);

	gtk_tree_view_append_column (treeview, column);
}

static void
init_treeview (GtkWidget *treeview)
{
	GtkTreeModel *model;

	/* the model */
	model = GTK_TREE_MODEL (gtk_list_store_new (NUM_COLS,
				G_TYPE_STRING,
				G_TYPE_STRING,
				G_TYPE_INT));

	/* the treeview */
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), model);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	g_object_unref (G_OBJECT (model));

	init_columns (GTK_TREE_VIEW (treeview));
	init_content (GTK_TREE_VIEW (treeview));
	gtk_widget_show (treeview);
}

static GtkWidget
*init_gui (void)
{
	GladeXML *xml;
	GtkWidget *window, *treeview, *checkbox, *exec_box, *exec;
	GdkPixbuf *pixbuf;

	xml = glade_xml_new (ACME_DATA "acme-properties.glade", NULL, NULL);

	/* the treeview */
	treeview = glade_xml_get_widget (xml, "treeview");
	init_treeview (treeview);

	window = glade_xml_get_widget (xml, "window");
	g_signal_connect (G_OBJECT (window), "response",
			G_CALLBACK (response_cb), NULL);

	pixbuf = gdk_pixbuf_new_from_file (ACME_DATA "keymap-48.png", NULL);
	if (pixbuf != NULL)
		gtk_window_set_icon (GTK_WINDOW (window), pixbuf);

	checkbox = glade_xml_get_widget (xml, "checkbutton1");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbox),
			gconf_client_get_bool (conf_client,
				"/apps/acme/use_pcm", NULL));
	g_signal_connect (G_OBJECT (checkbox), "toggled",
			G_CALLBACK (checkbox_toggled_cb), NULL);

	exec_box = glade_xml_get_widget (xml, "hbox2");
	gtk_widget_set_sensitive (exec_box, !is_running ());

	exec = glade_xml_get_widget (xml, "button1");
	g_signal_connect (GTK_BUTTON (exec), "clicked",
			G_CALLBACK (exec_clicked), exec_box);

	gtk_widget_show_all (window);

	return window;
}

int
main (int argc, char *argv[])
{
	GtkWidget *dialog;

	gnome_program_init ("acme-properties", VERSION,
			LIBGNOMEUI_MODULE,
			argc, argv,
			NULL);

	glade_gnome_init ();

	conf_client = gconf_client_get_default ();
	gconf_client_add_dir (conf_client,
			"/apps/acme",
			GCONF_CLIENT_PRELOAD_ONELEVEL,
			NULL);

	dialog = init_gui ();
	gtk_main ();

	return 0;
}

