# This file contains a collection of tests for the procedures in the
# file tclCompExpr.c.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1997 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: compExpr.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

if {([catch {expr T1()} msg] == 1) && ($msg == {unknown math function "T1"})} {
    set gotT1 0
    puts "This application hasn't been compiled with the \"T1\" and"
    puts "\"T2\" math functions, so I'll skip some of the expr tests."
} else {
    set gotT1 1
}

catch {unset a}

test compExpr-1.1 {TclCompileExpr procedure, successful expr parse and compile} {
    expr 1+2
} 3
test compExpr-1.2 {TclCompileExpr procedure, error parsing expr} {
    list [catch {expr 1+2+} msg] $msg
} {1 {syntax error in expression "1+2+": premature end of expression}}
test compExpr-1.3 {TclCompileExpr procedure, error compiling expr} {
    list [catch {expr "foo(123)"} msg] $msg
} {1 {unknown math function "foo"}}
test compExpr-1.4 {TclCompileExpr procedure, expr has no operators} {
    set a {000123}
    expr {$a}
} 83

test compExpr-2.1 {CompileSubExpr procedure, TCL_TOKEN_WORD parse token} {
    catch {unset a}
    set a 27
    expr {"foo$a" < "bar"}
} 0
test compExpr-2.2 {CompileSubExpr procedure, error compiling TCL_TOKEN_WORD parse token} {
    list [catch {expr {"00[expr 1+]" + 17}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}
test compExpr-2.3 {CompileSubExpr procedure, TCL_TOKEN_TEXT parse token} {
    expr {{12345}}
} 12345
test compExpr-2.4 {CompileSubExpr procedure, empty TCL_TOKEN_TEXT parse token} {
    expr {{}}
} {}
test compExpr-2.5 {CompileSubExpr procedure, TCL_TOKEN_BS parse token} {
    expr "\{  \\
 +123 \}"
} 123
test compExpr-2.6 {CompileSubExpr procedure, TCL_TOKEN_COMMAND parse token} {
    expr {[info tclversion] != ""}
} 1
test compExpr-2.7 {CompileSubExpr procedure, TCL_TOKEN_COMMAND parse token} {
    expr {[]}
} {}
test compExpr-2.8 {CompileSubExpr procedure, error in TCL_TOKEN_COMMAND parse token} {
    list [catch {expr {[foo "bar"xxx] + 17}} msg] $msg
} {1 {extra characters after close-quote}}
test compExpr-2.9 {CompileSubExpr procedure, TCL_TOKEN_VARIABLE parse token} {
    catch {unset a}
    set a 123
    expr {$a*2}
} 246
test compExpr-2.10 {CompileSubExpr procedure, TCL_TOKEN_VARIABLE parse token} {
    catch {unset a}
    catch {unset b}
    set a(george) martha
    set b geo
    expr {$a(${b}rge)}
} martha
test compExpr-2.11 {CompileSubExpr procedure, error in TCL_TOKEN_VARIABLE parse token} {
    catch {unset a}
    list [catch {expr {$a + 17}} msg] $msg
} {1 {can't read "a": no such variable}}
test compExpr-2.12 {CompileSubExpr procedure, TCL_TOKEN_SUB_EXPR parse token} {
    expr {27||3? 3<<(1+4) : 4&&9}
} 96
test compExpr-2.13 {CompileSubExpr procedure, error in TCL_TOKEN_SUB_EXPR parse token} {
    catch {unset a}
    set a 15
    list [catch {expr {27 || "$a[expr 1+]00"}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}
test compExpr-2.14 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, op found} {
    expr {5*6}
} 30
test compExpr-2.15 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, math function found} {
    format %.6g [expr {sin(2.0)}]
} 0.909297
test compExpr-2.16 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, math function not found} {
    list [catch {expr {fred(2.0)}} msg] $msg
} {1 {unknown math function "fred"}}
test compExpr-2.17 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4*2}
} 8
test compExpr-2.18 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4/2}
} 2
test compExpr-2.19 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4%2}
} 0
test compExpr-2.20 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4<<2}
} 16
test compExpr-2.21 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4>>2}
} 1
test compExpr-2.22 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4<2}
} 0
test compExpr-2.23 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4>2}
} 1
test compExpr-2.24 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4<=2}
} 0
test compExpr-2.25 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4>=2}
} 1
test compExpr-2.26 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4==2}
} 0
test compExpr-2.27 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4!=2}
} 1
test compExpr-2.28 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4&2}
} 0
test compExpr-2.29 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4^2}
} 6
test compExpr-2.30 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator} {
    expr {4|2}
} 6
test compExpr-2.31 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator, 1 operand} {
    expr {!4}
} 0
test compExpr-2.32 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator, 1 operand} {
    expr {~4}
} -5
test compExpr-2.33 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, normal operator, comparison} {
    catch {unset a}
    set a 15
    expr {$a==15}  ;# compiled out-of-line to runtime call on Tcl_ExprObjCmd
} 1
test compExpr-2.34 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    expr {+2}
} 2
test compExpr-2.35 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, error in special operator} {
    list [catch {expr {+[expr 1+]}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}
test compExpr-2.36 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    expr {4+2}
} 6
test compExpr-2.37 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, error in special operator} {
    list [catch {expr {[expr 1+]+5}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}
test compExpr-2.38 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, error in special operator} {
    list [catch {expr {5+[expr 1+]}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}
test compExpr-2.39 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    expr {-2}
} -2
test compExpr-2.40 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    expr {4-2}
} 2
test compExpr-2.41 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    catch {unset a}
    set a true
    expr {0||$a}
} 1
test compExpr-2.42 {CompileSubExpr procedure, error in TCL_TOKEN_SUB_EXPR parse token} {
    catch {unset a}
    set a 15
    list [catch {expr {27 || "$a[expr 1+]00"}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}
test compExpr-2.43 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    catch {unset a}
    set a false
    expr {3&&$a}
} 0
test compExpr-2.44 {CompileSubExpr procedure, TCL_TOKEN_OPERATOR token, special operator} {
    catch {unset a}
    set a false
    expr {$a||1? 1 : 0}
} 1
test compExpr-2.45 {CompileSubExpr procedure, error in TCL_TOKEN_SUB_EXPR parse token} {
    catch {unset a}
    set a 15
    list [catch {expr {1? 54 : "$a[expr 1+]00"}} msg] $msg
} {1 {syntax error in expression "1+": premature end of expression}}

test compExpr-3.1 {CompileLandOrLorExpr procedure, numeric 1st operand} {
    catch {unset a}
    set a 2
    expr {[set a]||0}
} 1
test compExpr-3.2 {CompileLandOrLorExpr procedure, nonnumeric 1st operand} {
    catch {unset a}
    set a no
    expr {$a&&1}
} 0
test compExpr-3.3 {CompileSubExpr procedure, error in 1st operand} {
    list [catch {expr {[expr *2]||0}} msg] $msg
} {1 {syntax error in expression "*2": unexpected operator *}}
test compExpr-3.4 {CompileLandOrLorExpr procedure, result is 1 or 0} {
    catch {unset a}
    catch {unset b}
    set a no
    set b true
    expr {$a || $b}
} 1
test compExpr-3.5 {CompileLandOrLorExpr procedure, short-circuit semantics} {
    catch {unset a}
    set a yes
    expr {$a || [exit]}
} 1
test compExpr-3.6 {CompileLandOrLorExpr procedure, short-circuit semantics} {
    catch {unset a}
    set a no
    expr {$a && [exit]}
} 0
test compExpr-3.7 {CompileLandOrLorExpr procedure, numeric 2nd operand} {
    catch {unset a}
    set a 2
    expr {0||[set a]}
} 1
test compExpr-3.8 {CompileLandOrLorExpr procedure, nonnumeric 2nd operand} {
    catch {unset a}
    set a no
    expr {1&&$a}
} 0
test compExpr-3.9 {CompileLandOrLorExpr procedure, error in 2nd operand} {
    list [catch {expr {0||[expr %2]}} msg] $msg
} {1 {syntax error in expression "%2": unexpected operator %}}
test compExpr-3.10 {CompileLandOrLorExpr procedure, long lor/land arm} {
    set a "abcdefghijkl"
    set i 7
    expr {[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]] || [string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]] || [string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]] || [string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]&&[string compare [format %c $i] [string index $a $i]]}
} 1

test compExpr-4.1 {CompileCondExpr procedure, simple test} {
    catch {unset a}
    set a 2
    expr {($a > 1)? "ok" : "nope"}
} ok
test compExpr-4.2 {CompileCondExpr procedure, complex test, convert to numeric} {
    catch {unset a}
    set a no
    expr {[set a]? 27 : -54}
} -54
test compExpr-4.3 {CompileCondExpr procedure, error in test} {
    list [catch {expr {[expr *2]? +1 : -1}} msg] $msg
} {1 {syntax error in expression "*2": unexpected operator *}}
test compExpr-4.4 {CompileCondExpr procedure, simple "true" clause} {
    catch {unset a}
    set a no
    expr {1? (27-2) : -54}
} 25
test compExpr-4.5 {CompileCondExpr procedure, convert "true" clause to numeric} {
    catch {unset a}
    set a no
    expr {1? $a : -54}
} no
test compExpr-4.6 {CompileCondExpr procedure, error in "true" clause} {
    list [catch {expr {1? [expr *2] : -127}} msg] $msg
} {1 {syntax error in expression "*2": unexpected operator *}}
test compExpr-4.7 {CompileCondExpr procedure, simple "false" clause} {
    catch {unset a}
    set a no
    expr {(2-2)? -3.14159 : "nope"}
} nope
test compExpr-4.8 {CompileCondExpr procedure, convert "false" clause to numeric} {
    catch {unset a}
    set a 00123
    expr {0? 42 : $a}
} 83
test compExpr-4.9 {CompileCondExpr procedure, error in "false" clause} {
    list [catch {expr {1? 15 : [expr *2]}} msg] $msg
} {1 {syntax error in expression "*2": unexpected operator *}}

test compExpr-5.1 {CompileMathFuncCall procedure, math function found} {
    format %.6g [expr atan2(1.0, 2.0)]
} 0.463648
test compExpr-5.2 {CompileMathFuncCall procedure, math function not found} {
    list [catch {expr {do_it()}} msg] $msg
} {1 {unknown math function "do_it"}}
if $gotT1 {
    test compExpr-5.3 {CompileMathFuncCall: call registered math function} {
	expr 3*T1()-1
    } 368
    test compExpr-5.4 {CompileMathFuncCall: call registered math function} {
	expr T2()*3
    } 1035
}
test compExpr-5.5 {CompileMathFuncCall procedure, too few arguments} {
    list [catch {expr {atan2(1.0)}} msg] $msg
} {1 {too few arguments for math function}}
test compExpr-5.6 {CompileMathFuncCall procedure, complex argument} {
    format %.6g [expr pow(2.1, 27.5-(24.4*(5%2)))]
} 9.97424
test compExpr-5.7 {CompileMathFuncCall procedure, error in argument} {
    list [catch {expr {sinh(2.*)}} msg] $msg
} {1 {syntax error in expression "sinh(2.*)": unexpected close parenthesis}}
test compExpr-5.8 {CompileMathFuncCall procedure, too many arguments} {
    list [catch {expr {sinh(2.0, 3.0)}} msg] $msg
} {1 {too many arguments for math function}}
test compExpr-5.9 {CompileMathFuncCall procedure, too many arguments} {
    list [catch {expr {0 <= rand(5.2)}} msg] $msg
} {1 {too many arguments for math function}}

test compExpr-6.1 {LogSyntaxError procedure, error in expr longer than 60 chars} {
    list [catch {expr {(+0123456)*(+0123456)*(+0123456)*(+0123456)*(+0123456)*(+0123456)*(+0123456)/} -1 foo 3} msg] $msg
} {1 {syntax error in expression "(+0123456)*(+0123456)*(+0123456)*(+0123456)*(+0123456)*(+012...": extra tokens at end of expression}}

# cleanup
catch {unset a}
catch {unset b}
::tcltest::cleanupTests
return
