(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "../cssLanguageTypes"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.getBrowserLabel = exports.textToMarkedString = exports.getEntryDescription = exports.browserNames = void 0;
    var cssLanguageTypes_1 = require("../cssLanguageTypes");
    exports.browserNames = {
        E: 'Edge',
        FF: 'Firefox',
        S: 'Safari',
        C: 'Chrome',
        IE: 'IE',
        O: 'Opera'
    };
    function getEntryStatus(status) {
        switch (status) {
            case 'experimental':
                return '⚠️ Property is experimental. Be cautious when using it.️\n\n';
            case 'nonstandard':
                return '🚨️ Property is nonstandard. Avoid using it.\n\n';
            case 'obsolete':
                return '🚨️️️ Property is obsolete. Avoid using it.\n\n';
            default:
                return '';
        }
    }
    function getEntryDescription(entry, doesSupportMarkdown, settings) {
        var result;
        if (doesSupportMarkdown) {
            result = {
                kind: 'markdown',
                value: getEntryMarkdownDescription(entry, settings)
            };
        }
        else {
            result = {
                kind: 'plaintext',
                value: getEntryStringDescription(entry, settings)
            };
        }
        if (result.value === '') {
            return undefined;
        }
        return result;
    }
    exports.getEntryDescription = getEntryDescription;
    function textToMarkedString(text) {
        text = text.replace(/[\\`*_{}[\]()#+\-.!]/g, '\\$&'); // escape markdown syntax tokens: http://daringfireball.net/projects/markdown/syntax#backslash
        return text.replace(/</g, '&lt;').replace(/>/g, '&gt;');
    }
    exports.textToMarkedString = textToMarkedString;
    function getEntryStringDescription(entry, settings) {
        if (!entry.description || entry.description === '') {
            return '';
        }
        if (typeof entry.description !== 'string') {
            return entry.description.value;
        }
        var result = '';
        if ((settings === null || settings === void 0 ? void 0 : settings.documentation) !== false) {
            if (entry.status) {
                result += getEntryStatus(entry.status);
            }
            result += entry.description;
            var browserLabel = getBrowserLabel(entry.browsers);
            if (browserLabel) {
                result += '\n(' + browserLabel + ')';
            }
            if ('syntax' in entry) {
                result += "\n\nSyntax: ".concat(entry.syntax);
            }
        }
        if (entry.references && entry.references.length > 0 && (settings === null || settings === void 0 ? void 0 : settings.references) !== false) {
            if (result.length > 0) {
                result += '\n\n';
            }
            result += entry.references.map(function (r) {
                return "".concat(r.name, ": ").concat(r.url);
            }).join(' | ');
        }
        return result;
    }
    function getEntryMarkdownDescription(entry, settings) {
        if (!entry.description || entry.description === '') {
            return '';
        }
        var result = '';
        if ((settings === null || settings === void 0 ? void 0 : settings.documentation) !== false) {
            if (entry.status) {
                result += getEntryStatus(entry.status);
            }
            if (typeof entry.description === 'string') {
                result += textToMarkedString(entry.description);
            }
            else {
                result += entry.description.kind === cssLanguageTypes_1.MarkupKind.Markdown ? entry.description.value : textToMarkedString(entry.description.value);
            }
            var browserLabel = getBrowserLabel(entry.browsers);
            if (browserLabel) {
                result += '\n\n(' + textToMarkedString(browserLabel) + ')';
            }
            if ('syntax' in entry && entry.syntax) {
                result += "\n\nSyntax: ".concat(textToMarkedString(entry.syntax));
            }
        }
        if (entry.references && entry.references.length > 0 && (settings === null || settings === void 0 ? void 0 : settings.references) !== false) {
            if (result.length > 0) {
                result += '\n\n';
            }
            result += entry.references.map(function (r) {
                return "[".concat(r.name, "](").concat(r.url, ")");
            }).join(' | ');
        }
        return result;
    }
    /**
     * Input is like `["E12","FF49","C47","IE","O"]`
     * Output is like `Edge 12, Firefox 49, Chrome 47, IE, Opera`
     */
    function getBrowserLabel(browsers) {
        if (browsers === void 0) { browsers = []; }
        if (browsers.length === 0) {
            return null;
        }
        return browsers
            .map(function (b) {
            var result = '';
            var matches = b.match(/([A-Z]+)(\d+)?/);
            var name = matches[1];
            var version = matches[2];
            if (name in exports.browserNames) {
                result += exports.browserNames[name];
            }
            if (version) {
                result += ' ' + version;
            }
            return result;
        })
            .join(', ');
    }
    exports.getBrowserLabel = getBrowserLabel;
});
