"use strict";
/*********************************************************************
 * Copyright (c) 2019 Arm and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const chai_1 = require("chai");
const utils_1 = require("./utils");
describe('breakpoints', () => __awaiter(void 0, void 0, void 0, function* () {
    let dc;
    beforeEach(() => __awaiter(void 0, void 0, void 0, function* () {
        dc = yield utils_1.standardBeforeEach();
        yield dc.launchRequest({
            verbose: true,
            gdb: utils_1.gdbPath,
            program: path.join(utils_1.testProgramsDir, 'count'),
            openGdbConsole: utils_1.openGdbConsole,
            logFile: '/tmp/log',
        });
    }));
    afterEach(() => __awaiter(void 0, void 0, void 0, function* () {
        yield dc.stop();
    }));
    it('hits a standard breakpoint', () => __awaiter(void 0, void 0, void 0, function* () {
        const bpResp = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                },
            ],
        });
        chai_1.expect(bpResp.body.breakpoints.length).eq(1);
        chai_1.expect(bpResp.body.breakpoints[0].verified).eq(true);
        chai_1.expect(bpResp.body.breakpoints[0].message).eq(undefined);
        yield dc.configurationDoneRequest();
        const scope = yield utils_1.getScopes(dc);
        const vr = scope.scopes.body.scopes[0].variablesReference;
        const vars = yield dc.variablesRequest({ variablesReference: vr });
        utils_1.verifyVariable(vars.body.variables[0], 'count', 'int', '0');
    }));
    it('handles breakpoints in multiple files', () => __awaiter(void 0, void 0, void 0, function* () {
        yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                },
            ],
        });
        yield dc.setBreakpointsRequest({
            source: {
                name: 'count_other.c',
                path: path.join(utils_1.testProgramsDir, 'count_other.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 2,
                },
            ],
        });
        yield dc.configurationDoneRequest();
        const scope = yield utils_1.getScopes(dc);
        const vr = scope.scopes.body.scopes[0].variablesReference;
        const vars = yield dc.variablesRequest({ variablesReference: vr });
        utils_1.verifyVariable(vars.body.variables[0], 'count', 'int', '0');
    }));
    it('fails gracefully on breakpoint on unknown file', () => __awaiter(void 0, void 0, void 0, function* () {
        const bpResp = yield dc.setBreakpointsRequest({
            source: {
                name: 'countBAD.c',
                path: path.join(utils_1.testProgramsDir, 'countBAD.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                },
            ],
        });
        chai_1.expect(bpResp.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp.body.breakpoints[0].verified).to.eq(false);
        chai_1.expect(bpResp.body.breakpoints[0].message).not.eq(undefined);
    }));
    it('fails gracefully on breakpoint on bad line in otherwise good source', () => __awaiter(void 0, void 0, void 0, function* () {
        const bpResp = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4 + 100000000,
                },
            ],
        });
        chai_1.expect(bpResp.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp.body.breakpoints[0].verified).to.eq(false);
        chai_1.expect(bpResp.body.breakpoints[0].message).not.eq(undefined);
    }));
    it('maintains breakpoint order when modifying breakpoints in a file', () => __awaiter(void 0, void 0, void 0, function* () {
        const bpResp1 = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 6,
                },
            ],
        });
        chai_1.expect(bpResp1.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp1.body.breakpoints[0].line).eq(6);
        const bpResp2 = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                },
                {
                    column: 1,
                    line: 6,
                },
            ],
        });
        chai_1.expect(bpResp2.body.breakpoints.length).to.eq(2);
        chai_1.expect(bpResp2.body.breakpoints[0].line).eq(4);
        chai_1.expect(bpResp2.body.breakpoints[1].line).eq(6);
        // Make sure the GDB id of the breakpoint on line 6 is unchanged
        chai_1.expect(bpResp2.body.breakpoints[1].id).eq(bpResp1.body.breakpoints[0].id);
    }));
    // Pending support for testing multiple GDB versions - test requires
    // GDB >= 8.2
    it.skip('reports back relocated line number', () => __awaiter(void 0, void 0, void 0, function* () {
        const args = {
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 5,
                },
            ],
        };
        const bpResp = yield dc.setBreakpointsRequest(args);
        // Make sure the GDB id of the breakpoint is unchanged
        chai_1.expect(bpResp.body.breakpoints[0].line).eq(6);
    }));
    // Pending support for testing multiple GDB versions - test requires
    // GDB >= 8.2
    it.skip('maintains gdb breakpoint when relocated', () => __awaiter(void 0, void 0, void 0, function* () {
        const args = {
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 5,
                },
            ],
        };
        const bpResp1 = yield dc.setBreakpointsRequest(args);
        chai_1.expect(bpResp1.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp1.body.breakpoints[0].line).eq(6);
        const bpResp2 = yield dc.setBreakpointsRequest(args);
        chai_1.expect(bpResp2.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp2.body.breakpoints[0].line).eq(6);
        // Make sure the GDB id of the breakpoint is unchanged
        chai_1.expect(bpResp2.body.breakpoints[0].id).eq(bpResp1.body.breakpoints[0].id);
    }));
    // Pending support for testing multiple GDB versions - test requires
    // GDB >= 8.2
    it.skip('maintains gdb breakpoint when relocated - files with spaces', () => __awaiter(void 0, void 0, void 0, function* () {
        const args = {
            source: {
                name: 'count space.c',
                path: path.join(utils_1.testProgramsDir, 'count space.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 7,
                },
            ],
        };
        const bpResp1 = yield dc.setBreakpointsRequest(args);
        chai_1.expect(bpResp1.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp1.body.breakpoints[0].line).eq(9);
        const bpResp2 = yield dc.setBreakpointsRequest(args);
        chai_1.expect(bpResp2.body.breakpoints.length).to.eq(1);
        chai_1.expect(bpResp2.body.breakpoints[0].line).eq(9);
        // Make sure the GDB id of the breakpoint is unchanged
        chai_1.expect(bpResp2.body.breakpoints[0].id).eq(bpResp1.body.breakpoints[0].id);
    }));
    it('hits a conditional breakpoint', () => __awaiter(void 0, void 0, void 0, function* () {
        yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                    condition: 'count == 5',
                },
            ],
        });
        yield dc.configurationDoneRequest();
        const scope = yield utils_1.getScopes(dc);
        const vr = scope.scopes.body.scopes[0].variablesReference;
        const vars = yield dc.variablesRequest({ variablesReference: vr });
        utils_1.verifyVariable(vars.body.variables[0], 'count', 'int', '5');
    }));
    it('hits a hit conditional breakpoint with >', () => __awaiter(void 0, void 0, void 0, function* () {
        yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                    hitCondition: '> 5',
                },
            ],
        });
        yield dc.configurationDoneRequest();
        const scope = yield utils_1.getScopes(dc);
        const vr = scope.scopes.body.scopes[0].variablesReference;
        const vars = yield dc.variablesRequest({ variablesReference: vr });
        utils_1.verifyVariable(vars.body.variables[0], 'count', 'int', '5');
    }));
    it('hits a hit conditional breakpoint without >', () => __awaiter(void 0, void 0, void 0, function* () {
        yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 4,
                    hitCondition: '5',
                },
            ],
        });
        yield dc.configurationDoneRequest();
        const scope = yield utils_1.getScopes(dc);
        const vr = scope.scopes.body.scopes[0].variablesReference;
        const vars = yield dc.variablesRequest({ variablesReference: vr });
        utils_1.verifyVariable(vars.body.variables[0], 'count', 'int', '4');
    }));
    it('resolves breakpoints', () => __awaiter(void 0, void 0, void 0, function* () {
        let response = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 2,
                },
            ],
        });
        chai_1.expect(response.body.breakpoints.length).to.eq(1);
        yield dc.configurationDoneRequest();
        yield dc.waitForEvent('stopped');
        response = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 2,
                },
                {
                    column: 1,
                    line: 3,
                },
            ],
        });
        chai_1.expect(response.body.breakpoints.length).to.eq(2);
        response = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 2,
                    condition: 'count == 5',
                },
                {
                    column: 1,
                    line: 3,
                },
            ],
        });
        chai_1.expect(response.body.breakpoints.length).to.eq(2);
        response = yield dc.setBreakpointsRequest({
            source: {
                name: 'count.c',
                path: path.join(utils_1.testProgramsDir, 'count.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: 2,
                    condition: 'count == 3',
                },
            ],
        });
        chai_1.expect(response.body.breakpoints.length).to.eq(1);
    }));
}));
//# sourceMappingURL=breakpoints.spec.js.map