"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const vscode_languageserver_1 = require("vscode-languageserver");
const yamlFolding_1 = require("../src/languageservice/services/yamlFolding");
const testHelper_1 = require("./utils/testHelper");
const context = { rangeLimit: 100000 };
suite('YAML Folding', () => {
    it('should return undefined if no document provided', () => {
        const ranges = yamlFolding_1.getFoldingRanges(undefined, context);
        chai_1.expect(ranges).to.be.undefined;
    });
    it('should return empty array for empty document', () => {
        const doc = testHelper_1.setupTextDocument('');
        const ranges = yamlFolding_1.getFoldingRanges(doc, context);
        chai_1.expect(ranges).to.be.empty;
    });
    it('should provide folding ranges for object', () => {
        const yaml = `
    foo: bar
    aaa:
      bbb: ccc
    `;
        const doc = testHelper_1.setupTextDocument(yaml);
        const ranges = yamlFolding_1.getFoldingRanges(doc, context);
        chai_1.expect(ranges.length).to.equal(1);
        chai_1.expect(ranges[0]).to.be.eql(vscode_languageserver_1.FoldingRange.create(2, 3, 4, 14));
    });
    it('should provide folding ranges for array', () => {
        const yaml = `
    foo: bar
    aaa:
      - bbb
    ccc: ddd
    `;
        const doc = testHelper_1.setupTextDocument(yaml);
        const ranges = yamlFolding_1.getFoldingRanges(doc, context);
        chai_1.expect(ranges.length).to.equal(1);
        chai_1.expect(ranges[0]).to.be.eql(vscode_languageserver_1.FoldingRange.create(2, 3, 4, 11));
    });
    it('should provide folding ranges for mapping in array', () => {
        const yaml = `
    foo: bar
    aaa:
      - bbb: "bbb"
        fff: "fff"
    ccc: ddd
    `;
        const doc = testHelper_1.setupTextDocument(yaml);
        const ranges = yamlFolding_1.getFoldingRanges(doc, context);
        chai_1.expect(ranges).to.deep.include.members([vscode_languageserver_1.FoldingRange.create(2, 4, 4, 18), vscode_languageserver_1.FoldingRange.create(3, 4, 8, 18)]);
    });
    it('should provide folding ranges for mapping in mapping', () => {
        const yaml = `
    foo: bar
    aaa:
      bbb:
        fff: "fff"
    ccc: ddd
    `;
        const doc = testHelper_1.setupTextDocument(yaml);
        const ranges = yamlFolding_1.getFoldingRanges(doc, context);
        chai_1.expect(ranges).to.deep.include.members([vscode_languageserver_1.FoldingRange.create(2, 4, 4, 18), vscode_languageserver_1.FoldingRange.create(3, 4, 6, 18)]);
    });
});
//# sourceMappingURL=yamlFolding.test.js.map