/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.notifications;

import static org.apache.commons.lang3.StringUtils.defaultString;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.NOTIFY_RESPONSES;

import java.util.Collections;
import java.util.List;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEvent;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEventGroup;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.dialogs.UIUtils;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Browsers;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Formats;
import org.eclipse.epp.logging.aeri.core.IProblemState;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.mylyn.commons.ui.compatibility.CommonColors;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;

public class ServerResponsesNotification extends Notification {

    private ILogEventGroup group;
    private ImageRegistry registry;

    @Inject
    public ServerResponsesNotification(ILogEventGroup group, ImageRegistry registry, IEventBroker broker) {
        super(NOTIFY_RESPONSES, broker);
        this.group = group;
        this.registry = registry;
    }

    public Control createControl(Composite parent) {
        GridDataFactory asis = GridDataFactory.fillDefaults().align(SWT.FILL, SWT.TOP);
        GridDataFactory w400 = GridDataFactory.fillDefaults().hint(NotificationPopup.MAX_WIDTH, SWT.DEFAULT).grab(false, false)
                .align(SWT.FILL, SWT.TOP);
        GridDataFactory w360 = GridDataFactory.fillDefaults().hint(NotificationPopup.MAX_WIDTH - 30, SWT.DEFAULT).grab(false, false)
                .align(SWT.FILL, SWT.TOP);
        Composite container = new Composite(parent, SWT.NO_FOCUS);
        GridLayoutFactory.fillDefaults().extendedMargins(0, 0, 0, 0).applyTo(container);
        w400.applyTo(container);

        String labelText = getLabel();
        if (StringUtils.isNotBlank(labelText)) {
            final Link label = new Link(container, SWT.WRAP | SWT.NO_FOCUS);
            label.setText(labelText);
            label.setForeground(CommonColors.TEXT_QUOTED);
            w400.applyTo(label);
        }

        final Link description = new Link(container, SWT.WRAP | SWT.NO_FOCUS);
        description.setText(getDescription());
        w400.applyTo(description);

        Composite serverSections = new Composite(container, SWT.NO_FOCUS);
        GridLayoutFactory.fillDefaults().numColumns(2).equalWidth(false).applyTo(serverSections);

        for (ILogEvent event : group.getEvents()) {
            IServerDescriptor server = event.getServer();
            final IProblemState response = event.getResponse();
            Label icon = new Label(serverSections, SWT.NO_FOCUS);
            Image image = UIUtils.decorate(server, response.getStatus(), registry);
            icon.setImage(image);
            asis.applyTo(icon);

            Link message = new Link(serverSections, SWT.NO_FOCUS | SWT.WRAP);
            w360.applyTo(message);
            message.setText(defaultString(response.getMessage()));
            message.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    Browsers.openInExternalBrowser(e.text);
                }
            });
        }
        return container;
    }

    @Override
    public String getTitle() {
        return Formats.format(Messages.NOTIFICATION_TITLE_SERVER_RESPONSES, group.getEvents().size());
    }

    @Override
    public String getLabel() {
        return ""; //$NON-NLS-1$
    }

    @Override
    public String getDescription() {
        return Formats.format(Messages.NOTIFICATION_DESCRIPTION_SERVER_RESPONSES,
                StringUtils.abbreviate(group.getStatus().getMessage(), 80), group.getEvents().size());
    }

    @Override
    public List<NotificationAction> getActions() {
        return Collections.emptyList();
    }

}
