/**
 * Copyright (c) 2010-2016, Peter Lunk, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Peter Lunk - initial API and implementation
 */
package org.eclipse.viatra.query.testing.core.api;

import org.eclipse.viatra.query.testing.snapshot.SerializedJavaObjectSubstitution;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Inheriting classes should implement the abstract methods of this class, allowing the VIATRA
 * Query Testing Framework to access and compare plain java typed objects as well.
 * 
 * @author Peter Lunk
 */
@Data
@SuppressWarnings("all")
public abstract class JavaObjectAccess {
  private final Class<?> type;
  
  /**
   * Constructs a {@link SerializedJavaObjectSubstitution} element based on the provided
   * object. Implementation should provide a serialized representation of the given object.
   * This is called when a match set is converted to a snapshot.
   * 
   * @since 1.6
   */
  public abstract SerializedJavaObjectSubstitution toSubstitution(final Object value);
  
  /**
   * Calculates the hash code for a given {@link SerializedJavaObjectSubstitution}. Note, that if
   * two {@link SerializedJavaObjectSubstitution} elements are equal, their hash code should be the same as well.
   * 
   * @since 1.6
   */
  public abstract int calculateHash(final SerializedJavaObjectSubstitution substitution);
  
  /**
   * Returns if two {@link SerializedJavaObjectSubstitution} elements are equal or not.
   * Implementations should cover the deserialization of {@link SerializedJavaObjectSubstitution} contents.
   * 
   * @since 1.6
   */
  public abstract boolean equals(final SerializedJavaObjectSubstitution a, final SerializedJavaObjectSubstitution b);
  
  public JavaObjectAccess(final Class<?> type) {
    super();
    this.type = type;
  }
  
  @Override
  @Pure
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.type== null) ? 0 : this.type.hashCode());
    return result;
  }
  
  @Override
  @Pure
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    JavaObjectAccess other = (JavaObjectAccess) obj;
    if (this.type == null) {
      if (other.type != null)
        return false;
    } else if (!this.type.equals(other.type))
      return false;
    return true;
  }
  
  @Override
  @Pure
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("type", this.type);
    return b.toString();
  }
  
  @Pure
  public Class<?> getType() {
    return this.type;
  }
}
