/*******************************************************************************
 * Copyright (c) 2009 itemis AG (http://www.itemis.eu) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package org.eclipse.emf.emfindex.store;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.emfindex.EObjectDescriptor;
import org.eclipse.emf.emfindex.EReferenceDescriptor;
import org.eclipse.emf.emfindex.ResourceDescriptor;

/**
 * @author Jan Khnlein - Initial contribution and API
 */
public class ResourceIndexerImpl implements ResourceIndexer {

	public void resourceChanged(URI resourceURI, IndexUpdater updater) {
		Resource resource = loadResource(resourceURI);
		this.resourceChanged(resource, updater);
	}

	public void resourceDeleted(URI resourceURI, IndexUpdater updater) {
		updater.deleteResourceDescriptor(resourceURI);
	}

	protected Resource loadResource(URI resourceURI) {
		ResourceSet resourceSet = new ResourceSetImpl();
		return resourceSet.getResource(resourceURI, true);
	}

	public void resourceChanged(Resource resource, IndexUpdater updater) {
		if (resource != null) {
			DescriptorFactory descriptorFactory = updater
					.getDescriptorFactory();
			URIConverter uriConverter = getURIConverter(resource);
			ResourceDescriptor resourceDescriptor = createResourceDescriptor(
					resource, descriptorFactory, uriConverter);
			Map<EObject, EObjectDescriptor> eObject2DescriptorCache = new HashMap<EObject, EObjectDescriptor>();
			createEObjectDescriptors(resourceDescriptor, resource, resource
					.getContents(), uriConverter, descriptorFactory,
					eObject2DescriptorCache);
			createEReferenceDescriptors(resourceDescriptor, resource, resource
					.getContents(), uriConverter, descriptorFactory,
					eObject2DescriptorCache);
			updater.update(resourceDescriptor);
		}
	}

	protected void createEObjectDescriptors(
			ResourceDescriptor containerDescriptor, Resource resource,
			EList<EObject> contents, URIConverter uriConverter,
			DescriptorFactory descriptorFactory,
			Map<EObject, EObjectDescriptor> eObject2DescriptorCache) {
		for (EObject eObject : contents) {
			if (isIndexElement(eObject, resource)) {
				if (eObject.eResource() == resource) {
					EObjectDescriptor eObjectDescriptor = createEObjectDescriptor(
							eObject, resource, containerDescriptor,
							descriptorFactory);
					eObject2DescriptorCache.put(eObject, eObjectDescriptor);
				}
			}
			if (isIndexChildren(eObject, resource)) {
				createEObjectDescriptors(containerDescriptor, resource, eObject
						.eContents(), uriConverter, descriptorFactory,
						eObject2DescriptorCache);
			}
		}
	}

	protected void createEReferenceDescriptors(
			ResourceDescriptor containerDescriptor, Resource resource,
			EList<EObject> contents, URIConverter uriConverter,
			DescriptorFactory descriptorFactory,
			Map<EObject, EObjectDescriptor> eObject2DescriptorCache) {
		for (EObject eObject : contents) {
			String sourceFragment = resource.getURIFragment(eObject);
			if (isIndexReferences(eObject, resource)) {
				for (EReference eReference : eObject.eClass()
						.getEAllReferences()) {
					if (isIndexReference(eReference, eObject)) {
						if (eReference.isMany()) {
							List<?> targets = (List<?>) eObject.eGet(
									eReference, false);
							for (int index = 0; index < targets.size(); ++index) {
								Object target = targets.get(index);
								createEReferenceDescriptor(sourceFragment,
										target, eReference, containerDescriptor,
										descriptorFactory, uriConverter,
										eObject2DescriptorCache);
							}
						} else {
							Object target = eObject.eGet(eReference, false);
							createEReferenceDescriptor(sourceFragment, target,
									eReference, containerDescriptor,
									descriptorFactory, uriConverter,
									eObject2DescriptorCache);
						}
					}
				}
			}
			if (isIndexChildren(eObject, resource)) {
				createEReferenceDescriptors(containerDescriptor, resource, eObject
						.eContents(), uriConverter, descriptorFactory,
						eObject2DescriptorCache);
			}
		}
	}

	protected URIConverter getURIConverter(Resource resource) {
		ResourceSet resourceSet = resource.getResourceSet();
		return (resourceSet != null) ? resourceSet.getURIConverter()
				: URIConverter.INSTANCE;
	}

	protected ResourceDescriptor createResourceDescriptor(Resource resource,
			DescriptorFactory descriptorFactory, URIConverter uriConverter) {
		ResourceDescriptor resourceDescriptor = descriptorFactory
				.createResourceDescriptor(null, uriConverter.normalize(resource
						.getURI()), System.currentTimeMillis());
		addResourceUserData(resource, resourceDescriptor, descriptorFactory);
		return resourceDescriptor;
	}

	protected EObjectDescriptor createEObjectDescriptor(EObject eObject,
			Resource resource, ResourceDescriptor containerDescriptor,
			DescriptorFactory descriptorFactory) {
		EObjectDescriptor eObjectDescriptor = descriptorFactory
				.createEObjectDescriptor(containerDescriptor,
						getEObjectName(eObject), resource
								.getURIFragment(eObject), eObject.eClass());
		addEObjectUserData(eObject, eObjectDescriptor, descriptorFactory);
		return eObjectDescriptor;
	}

	protected EReferenceDescriptor createEReferenceDescriptor(
			String sourceFragment, Object target, EReference eReference,
			ResourceDescriptor containerDescriptor,
			DescriptorFactory descriptorFactory, URIConverter uriConverter,
			Map<EObject, EObjectDescriptor> eObject2DescriptorCache) {
		if (target instanceof EObject) {
			EObjectDescriptor targetEObjectDescriptor = eObject2DescriptorCache
					.get(target);
			if (targetEObjectDescriptor != null) {
				return descriptorFactory.createEReferenceDescriptor(
						containerDescriptor, sourceFragment,
						targetEObjectDescriptor, eReference);
			} else {
				URI targetFragmentURI = uriConverter.normalize(EcoreUtil
						.getURI((EObject) target));
				return descriptorFactory.createEReferenceDescriptor(
						containerDescriptor, sourceFragment, targetFragmentURI,
						eReference);
			}
		}
		return null;
	}

	/**
	 * Subclasses can override this method to extract userData from the resource
	 * and add it to the {@link ResourceDescriptor} by means of
	 * {@link DescriptorFactory#addUserData(org.eclipse.emf.emfindex.BaseDescriptor, String, Serializable)}
	 * 
	 * @param resource
	 * @param resourceDescriptor
	 * @param descriptorFactory
	 */
	protected void addResourceUserData(Resource resource,
			ResourceDescriptor resourceDescriptor,
			DescriptorFactory descriptorFactory) {
	}

	/**
	 * Subclasses can override this method to extract userData from the
	 * {@link EObject} and add it to the {@link EObjectDescriptor} by means of
	 * {@link DescriptorFactory#addUserData(org.eclipse.emf.emfindex.BaseDescriptor, String, Serializable)}
	 * 
	 * @param resource
	 * @param resourceDescriptor
	 * @param descriptorFactory
	 */
	protected void addEObjectUserData(EObject eObject,
			EObjectDescriptor eObjectDescriptor,
			DescriptorFactory descriptorFactory) {
	}

	/**
	 * Subclasses can override this method to extract the name form an
	 * {@link EObject}.
	 * 
	 * @param eObject
	 * @return
	 */
	@SuppressWarnings("unchecked")
	protected String getEObjectName(EObject eObject) {
		EStructuralFeature nameFeature = eObject.eClass()
				.getEStructuralFeature("name");
		if (nameFeature != null && nameFeature.getEType() instanceof EDataType) {
			if (!nameFeature.isMany()) {
				Object nameFeatureValue = eObject.eGet(nameFeature);
				return (nameFeatureValue == null) ? null : nameFeatureValue
						.toString();
			} else {
				List names = (List) eObject.eGet(nameFeature);
				StringBuilder b = new StringBuilder();
				for (Iterator nameIter = names.iterator(); nameIter.hasNext();) {
					b.append(nameIter.next().toString());
				}
				return b.toString();
			}
		}
		return null;
	}

	protected boolean isIndexElement(EObject eObject, Resource resource) {
		return eObject.eResource() == resource;
	}

	protected boolean isIndexReferences(EObject eObject, Resource resource) {
		return true;
	}

	protected boolean isIndexReference(EReference eReference, EObject element) {
		return !eReference.isContainment() && !eReference.isDerived()
				&& element.eIsSet(eReference);
	}

	protected boolean isIndexChildren(EObject eObject, Resource resource) {
		return true;
	}

}
