/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.riot.thrift;

import java.math.BigDecimal;
import java.math.BigInteger;
import org.apache.jena.atlas.lib.Cache;
import org.apache.jena.atlas.lib.Pair;
import org.apache.jena.datatypes.RDFDatatype;
import org.apache.jena.datatypes.xsd.XSDDatatype;
import org.apache.jena.datatypes.xsd.impl.RDFLangString;
import org.apache.jena.graph.Node;
import org.apache.jena.graph.NodeFactory;
import org.apache.jena.graph.Triple;
import org.apache.jena.riot.system.PrefixMap;
import org.apache.jena.riot.system.PrefixMapFactory;
import org.apache.jena.riot.system.RiotLib;
import org.apache.jena.riot.thrift.RiotThriftException;
import org.apache.jena.riot.thrift.TRDF;
import org.apache.jena.riot.thrift.wire.RDF_BNode;
import org.apache.jena.riot.thrift.wire.RDF_Decimal;
import org.apache.jena.riot.thrift.wire.RDF_IRI;
import org.apache.jena.riot.thrift.wire.RDF_Literal;
import org.apache.jena.riot.thrift.wire.RDF_PrefixName;
import org.apache.jena.riot.thrift.wire.RDF_Quad;
import org.apache.jena.riot.thrift.wire.RDF_Term;
import org.apache.jena.riot.thrift.wire.RDF_Triple;
import org.apache.jena.riot.thrift.wire.RDF_VAR;
import org.apache.jena.sparql.core.Quad;
import org.apache.jena.sparql.core.Var;
import org.apache.thrift.TDeserializer;
import org.apache.thrift.TException;
import org.apache.thrift.TSerializer;
import org.apache.thrift.protocol.TCompactProtocol;

public class ThriftConvert {
    private static final PrefixMap emptyPrefixMap = PrefixMapFactory.emptyPrefixMap();
    static final BigInteger MAX_I = BigInteger.valueOf(Long.MAX_VALUE);
    static final BigInteger MIN_I = BigInteger.valueOf(Long.MIN_VALUE);

    public static boolean toThriftValue(Node node, RDF_Term term) {
        if (!node.isLiteral()) {
            return false;
        }
        String lex = node.getLiteralLexicalForm();
        RDFDatatype rdt = node.getLiteralDatatype();
        if (rdt == null) {
            return false;
        }
        if (rdt.equals(XSDDatatype.XSDdecimal)) {
            if (rdt.isValid(lex)) {
                BigDecimal decimal = new BigDecimal(lex.trim());
                int scale = decimal.scale();
                BigInteger bigInt = decimal.unscaledValue();
                if (bigInt.compareTo(MAX_I) <= 0 && bigInt.compareTo(MIN_I) >= 0) {
                    RDF_Decimal d = new RDF_Decimal(bigInt.longValue(), scale);
                    RDF_Term literal = new RDF_Term();
                    term.setValDecimal(d);
                    return true;
                }
            }
        } else if (rdt.equals(XSDDatatype.XSDinteger) || rdt.equals(XSDDatatype.XSDlong) || rdt.equals(XSDDatatype.XSDint) || rdt.equals(XSDDatatype.XSDshort) || rdt.equals(XSDDatatype.XSDbyte)) {
            if (rdt.isValid(lex)) {
                try {
                    long v = ((Number)node.getLiteralValue()).longValue();
                    term.setValInteger(v);
                    return true;
                }
                catch (Throwable v) {}
            }
        } else if (rdt.equals(XSDDatatype.XSDdouble) && rdt.isValid(lex)) {
            try {
                double v = (Double)node.getLiteralValue();
                term.setValDouble(v);
                return true;
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
        return false;
    }

    public static void toThrift(Node node, RDF_Term term) {
        ThriftConvert.toThrift(node, emptyPrefixMap, term, true);
    }

    public static void toThrift(Node node, RDF_Term term, boolean allowValues) {
        ThriftConvert.toThrift(node, emptyPrefixMap, term, allowValues);
    }

    public static void toThrift(Node node, PrefixMap pmap, RDF_Term term, boolean allowValues) {
        RDF_PrefixName prefixName;
        if (node == null) {
            term.setUndefined(TRDF.UNDEF);
            return;
        }
        if (node.isURI() && (prefixName = ThriftConvert.abbrev(node.getURI(), pmap)) != null) {
            term.setPrefixName(prefixName);
            return;
        }
        if (node.isBlank()) {
            RDF_BNode b = new RDF_BNode(node.getBlankNodeLabel());
            term.setBnode(b);
            return;
        }
        if (node.isURI()) {
            RDF_IRI iri = new RDF_IRI(node.getURI());
            term.setIri(iri);
            return;
        }
        if (node.isLiteral()) {
            boolean b;
            if (allowValues && (b = ThriftConvert.toThriftValue(node, term))) {
                return;
            }
            String lex = node.getLiteralLexicalForm();
            String dt = node.getLiteralDatatypeURI();
            String lang = node.getLiteralLanguage();
            RDF_Literal literal = new RDF_Literal(lex);
            if (node.getLiteralDatatype().equals(XSDDatatype.XSDstring) || node.getLiteralDatatype().equals(RDFLangString.rdfLangString)) {
                dt = null;
            }
            if (dt != null) {
                RDF_PrefixName dtPrefixName = ThriftConvert.abbrev(dt, pmap);
                if (dtPrefixName != null) {
                    literal.setDtPrefix(dtPrefixName);
                } else {
                    literal.setDatatype(dt);
                }
            }
            if (lang != null && !lang.isEmpty()) {
                literal.setLangtag(lang);
            }
            term.setLiteral(literal);
            return;
        }
        if (node.isVariable()) {
            RDF_VAR var = new RDF_VAR(node.getName());
            term.setVariable(var);
            return;
        }
        if (node.isNodeTriple()) {
            Triple triple = node.getTriple();
            RDF_Term sTerm = new RDF_Term();
            ThriftConvert.toThrift(triple.getSubject(), pmap, sTerm, allowValues);
            RDF_Term pTerm = new RDF_Term();
            ThriftConvert.toThrift(triple.getPredicate(), pmap, pTerm, allowValues);
            RDF_Term oTerm = new RDF_Term();
            ThriftConvert.toThrift(triple.getObject(), pmap, oTerm, allowValues);
            RDF_Triple tripleTerm = new RDF_Triple(sTerm, pTerm, oTerm);
            term.setTripleTerm(tripleTerm);
            return;
        }
        if (Node.ANY.equals(node)) {
            term.setAny(TRDF.ANY);
            return;
        }
        throw new RiotThriftException("Node conversion not supported: " + node);
    }

    public static Node convert(RDF_Term term) {
        return ThriftConvert.convert(null, term, null);
    }

    public static Node convert(Cache<String, Node> uriCache, RDF_Term term) {
        return ThriftConvert.convert(uriCache, term, null);
    }

    public static Node convert(RDF_Term term, PrefixMap pmap) {
        return ThriftConvert.convert(null, term, pmap);
    }

    public static Node convert(Cache<String, Node> uriCache, RDF_Term term, PrefixMap pmap) {
        if (term.isSetPrefixName()) {
            String x = ThriftConvert.expand(term.getPrefixName(), pmap);
            if (x != null) {
                return ThriftConvert.uri(uriCache, x);
            }
            throw new RiotThriftException("Failed to expand " + term);
        }
        if (term.isSetIri()) {
            String s = term.getIri().getIri();
            return ThriftConvert.uri(uriCache, s);
        }
        if (term.isSetBnode()) {
            return NodeFactory.createBlankNode(term.getBnode().getLabel());
        }
        if (term.isSetLiteral()) {
            RDF_Literal lit = term.getLiteral();
            String lex = lit.getLex();
            String dtString = null;
            if (lit.isSetDatatype()) {
                dtString = lit.getDatatype();
            } else if (lit.isSetDtPrefix()) {
                String x = ThriftConvert.expand(lit.getDtPrefix(), pmap);
                if (x == null) {
                    throw new RiotThriftException("Failed to expand datatype prefix name:" + term);
                }
                dtString = x;
            }
            RDFDatatype dt = NodeFactory.getType(dtString);
            String lang = lit.getLangtag();
            return NodeFactory.createLiteral(lex, lang, dt);
        }
        if (term.isSetValInteger()) {
            long x = term.getValInteger();
            String lex = Long.toString(x, 10);
            XSDDatatype dt = XSDDatatype.XSDinteger;
            return NodeFactory.createLiteralDT(lex, dt);
        }
        if (term.isSetValDouble()) {
            double x = term.getValDouble();
            String lex = Double.toString(x);
            XSDDatatype dt = XSDDatatype.XSDdouble;
            return NodeFactory.createLiteralDT(lex, dt);
        }
        if (term.isSetValDecimal()) {
            long value = term.getValDecimal().getValue();
            int scale = term.getValDecimal().getScale();
            BigDecimal d = BigDecimal.valueOf(value, scale);
            String lex = d.toPlainString();
            XSDDatatype dt = XSDDatatype.XSDdecimal;
            return NodeFactory.createLiteralDT(lex, dt);
        }
        if (term.isSetTripleTerm()) {
            RDF_Triple rt = term.getTripleTerm();
            Triple t = ThriftConvert.convert(rt, pmap);
            return NodeFactory.createTripleNode(t);
        }
        if (term.isSetVariable()) {
            return Var.alloc(term.getVariable().getName());
        }
        if (term.isSetAny()) {
            return Node.ANY;
        }
        if (term.isSetUndefined()) {
            return null;
        }
        throw new RiotThriftException("No conversion to a Node: " + term.toString());
    }

    private static Node uri(Cache<String, Node> uriCache, String uriStr) {
        if (uriCache != null) {
            return uriCache.get(uriStr, RiotLib::createIRIorBNode);
        }
        return RiotLib.createIRIorBNode(uriStr);
    }

    private static String expand(RDF_PrefixName prefixName, PrefixMap pmap) {
        String localname2;
        if (pmap == null) {
            return null;
        }
        String prefix = prefixName.getPrefix();
        String x = pmap.expand(prefix, localname2 = prefixName.getLocalName());
        if (x == null) {
            throw new RiotThriftException("Failed to expand " + prefixName);
        }
        return x;
    }

    public static RDF_Term convert(Node node, boolean allowValues) {
        return ThriftConvert.convert(node, null, allowValues);
    }

    public static RDF_Term convert(Node node, PrefixMap pmap, boolean allowValues) {
        RDF_Term n = new RDF_Term();
        ThriftConvert.toThrift(node, pmap, n, allowValues);
        return n;
    }

    private static RDF_PrefixName abbrev(String uriStr, PrefixMap pmap) {
        if (pmap == null) {
            return null;
        }
        Pair<String, String> p = pmap.abbrev(uriStr);
        if (p == null) {
            return null;
        }
        return new RDF_PrefixName(p.getLeft(), p.getRight());
    }

    public static Triple convert(RDF_Triple triple) {
        return ThriftConvert.convert(triple, null);
    }

    public static Triple convert(RDF_Triple rt, PrefixMap pmap) {
        return ThriftConvert.convert(null, rt, pmap);
    }

    public static Triple convert(Cache<String, Node> uriCache, RDF_Triple rt, PrefixMap pmap) {
        Node s = ThriftConvert.convert(uriCache, rt.getS(), pmap);
        Node p = ThriftConvert.convert(uriCache, rt.getP(), pmap);
        Node o = ThriftConvert.convert(uriCache, rt.getO(), pmap);
        return Triple.create(s, p, o);
    }

    public static RDF_Triple convert(Triple triple, boolean allowValues) {
        return ThriftConvert.convert(triple, null, allowValues);
    }

    public static RDF_Triple convert(Triple triple, PrefixMap pmap, boolean allowValues) {
        RDF_Triple t = new RDF_Triple();
        RDF_Term s = ThriftConvert.convert(triple.getSubject(), pmap, allowValues);
        RDF_Term p = ThriftConvert.convert(triple.getPredicate(), pmap, allowValues);
        RDF_Term o = ThriftConvert.convert(triple.getObject(), pmap, allowValues);
        t.setS(s);
        t.setP(p);
        t.setO(o);
        return t;
    }

    public static Quad convert(RDF_Quad quad) {
        return ThriftConvert.convert(quad, null);
    }

    public static Quad convert(RDF_Quad rq, PrefixMap pmap) {
        return ThriftConvert.convert(null, rq, pmap);
    }

    public static Quad convert(Cache<String, Node> uriCache, RDF_Quad rq, PrefixMap pmap) {
        Node g = rq.isSetG() ? ThriftConvert.convert(uriCache, rq.getG(), pmap) : null;
        Node s = ThriftConvert.convert(uriCache, rq.getS(), pmap);
        Node p = ThriftConvert.convert(uriCache, rq.getP(), pmap);
        Node o = ThriftConvert.convert(uriCache, rq.getO(), pmap);
        return Quad.create(g, s, p, o);
    }

    public static RDF_Quad convert(Quad quad, boolean allowValues) {
        return ThriftConvert.convert(quad, null, allowValues);
    }

    public static RDF_Quad convert(Quad quad, PrefixMap pmap, boolean allowValues) {
        RDF_Quad q = new RDF_Quad();
        RDF_Term g = null;
        if (quad.getGraph() != null) {
            g = ThriftConvert.convert(quad.getGraph(), pmap, allowValues);
        }
        RDF_Term s = ThriftConvert.convert(quad.getSubject(), pmap, allowValues);
        RDF_Term p = ThriftConvert.convert(quad.getPredicate(), pmap, allowValues);
        RDF_Term o = ThriftConvert.convert(quad.getObject(), pmap, allowValues);
        if (g != null) {
            q.setG(g);
        }
        q.setS(s);
        q.setP(p);
        q.setO(o);
        return q;
    }

    public static byte[] termToBytes(RDF_Term term) {
        try {
            TSerializer serializer = new TSerializer(new TCompactProtocol.Factory());
            return serializer.serialize(term);
        }
        catch (TException e2) {
            throw new RiotThriftException(e2);
        }
    }

    public static RDF_Term termFromBytes(byte[] bytes) {
        RDF_Term term = new RDF_Term();
        ThriftConvert.termFromBytes(term, bytes);
        return term;
    }

    public static void termFromBytes(RDF_Term term, byte[] bytes) {
        try {
            TDeserializer deserializer = new TDeserializer(new TCompactProtocol.Factory());
            deserializer.deserialize(term, bytes);
        }
        catch (TException e2) {
            throw new RiotThriftException(e2);
        }
    }
}

