/**
 * Copyright (c) 2016 TypeFox GmbH (http://www.typefox.io) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.elk.graph.text.conversion;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.List;
import org.eclipse.elk.core.data.LayoutMetaDataService;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.core.util.internal.LayoutOptionProxy;
import org.eclipse.elk.graph.impl.ElkPropertyToValueMapEntryImpl;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.text.services.ElkGraphGrammarAccess;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.TerminalRule;
import org.eclipse.xtext.conversion.ValueConverterException;
import org.eclipse.xtext.conversion.impl.AbstractValueConverter;
import org.eclipse.xtext.conversion.impl.IDValueConverter;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class PropertyValueValueConverter extends AbstractValueConverter<Object> {
  private IDValueConverter idValueConverter;
  
  @Inject
  public void initialize(final Provider<IDValueConverter> idValueConverterProvider, final ElkGraphGrammarAccess grammarAccess) {
    IDValueConverter _get = idValueConverterProvider.get();
    final Procedure1<IDValueConverter> _function = new Procedure1<IDValueConverter>() {
      @Override
      public void apply(final IDValueConverter it) {
        TerminalRule _iDRule = grammarAccess.getIDRule();
        it.setRule(_iDRule);
      }
    };
    IDValueConverter _doubleArrow = ObjectExtensions.<IDValueConverter>operator_doubleArrow(_get, _function);
    this.idValueConverter = _doubleArrow;
  }
  
  @Override
  public String toString(final Object value) throws ValueConverterException {
    if ((value == null)) {
      throw new ValueConverterException("Value may not be null.", null, null);
    } else {
      if ((((value instanceof Boolean) || (value instanceof Number)) || (value instanceof Enum<?>))) {
        return value.toString();
      } else {
        String _string = value.toString();
        return this.quoteIfNecessary(_string);
      }
    }
  }
  
  private String quoteIfNecessary(final String s) {
    try {
      List<String> _split = Strings.split(s, ".");
      final Function1<String, String> _function = new Function1<String, String>() {
        @Override
        public String apply(final String it) {
          return PropertyValueValueConverter.this.idValueConverter.toString(it);
        }
      };
      List<String> _map = ListExtensions.<String, String>map(_split, _function);
      return IterableExtensions.join(_map, ".");
    } catch (final Throwable _t) {
      if (_t instanceof ValueConverterException) {
        final ValueConverterException e = (ValueConverterException)_t;
        return (("\"" + s) + "\"");
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Override
  public Object toValue(final String string, final INode node) throws ValueConverterException {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(string);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      final String unquotedString = this.unquoteIfNecessary(string);
      EObject _semanticElement = null;
      if (node!=null) {
        _semanticElement=node.getSemanticElement();
      }
      final EObject semanticElem = _semanticElement;
      if ((semanticElem instanceof ElkPropertyToValueMapEntryImpl)) {
        IProperty<?> _key = ((ElkPropertyToValueMapEntryImpl)semanticElem).getKey();
        final LayoutOptionData option = this.toLayoutOption(_key);
        Object _parseValue = null;
        if (option!=null) {
          _parseValue=option.parseValue(unquotedString);
        }
        final Object value = _parseValue;
        if ((value != null)) {
          return value;
        }
      }
      return new LayoutOptionProxy(unquotedString);
    }
    return null;
  }
  
  private String unquoteIfNecessary(final String s) {
    if ((((s.length() >= 2) && s.startsWith("\"")) && s.endsWith("\""))) {
      int _length = s.length();
      int _minus = (_length - 1);
      return s.substring(1, _minus);
    } else {
      if (((s.length() >= 1) && (Character.isJavaIdentifierStart(s.charAt(0)) || s.startsWith("^")))) {
        List<String> _split = Strings.split(s, ".");
        final Function1<String, String> _function = new Function1<String, String>() {
          @Override
          public String apply(final String it) {
            return PropertyValueValueConverter.this.idValueConverter.toValue(it, null);
          }
        };
        List<String> _map = ListExtensions.<String, String>map(_split, _function);
        return IterableExtensions.join(_map, ".");
      } else {
        return s;
      }
    }
  }
  
  private LayoutOptionData toLayoutOption(final IProperty<?> property) {
    if ((property instanceof LayoutOptionData)) {
      return ((LayoutOptionData)property);
    } else {
      if ((property != null)) {
        LayoutMetaDataService _instance = LayoutMetaDataService.getInstance();
        String _id = property.getId();
        return _instance.getOptionData(_id);
      }
    }
    return null;
  }
}
