/**
 * <copyright>
 * 
 * Copyright (c) 2009-2010 Thales Corporate Services S.A.S.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Thales Corporate Services S.A.S - initial API and implementation
 * 
 * </copyright>
 */

package org.eclipse.egf.pattern.ecore;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.egf.core.EGFCorePlugin;
import org.eclipse.egf.core.domain.EGFResourceSet;
import org.eclipse.egf.core.genmodel.IPlatformGenModel;
import org.eclipse.egf.core.platform.EGFPlatformPlugin;
import org.eclipse.egf.core.platform.pde.IPlatformExtensionPointDelta;
import org.eclipse.egf.core.platform.pde.IPlatformExtensionPointListener;
import org.eclipse.egf.pattern.EGFPatternPlugin;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.WrappedException;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.impl.EPackageRegistryImpl;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.osgi.util.NLS;

/**
 * The purpose is to handle ecore models from the workspace as well as runtime
 * ones.
 * 
 * TODO Une fois terminée, cette classe devrait plutôt se trouver dans core ou
 * platform <br>
 * TODO revoir l'utilisation de ProjectClassLoaderHelper <br>
 * 
 * @author Thomas Guiu
 * 
 */
public class EPackageHelper {

    private static final EPackageHelper INSTANCE = new EPackageHelper();

    public static final String INSTANCE_FIELD_NAME = "eINSTANCE"; //$NON-NLS-1$

    // TODO: we mix the target definition with the runtime definition
    // This should be clarified and processed

    public final EPackage.Registry REGISTRY = new EPackageRegistryImpl(EPackage.Registry.INSTANCE);

    private Map<String, String> nsuri2basePackage = null;

    private IPlatformExtensionPointListener _extensionPointListener = new IPlatformExtensionPointListener() {

        public void platformExtensionPointChanged(IPlatformExtensionPointDelta delta) {
            if (nsuri2basePackage == null) {
                init();
            }
            for (IPlatformGenModel genModel : delta.getAddedPlatformExtensionPoints(IPlatformGenModel.class)) {
                addEcoreModel(genModel);
            }
            for (IPlatformGenModel genModel : delta.getRemovedPlatformExtensionPoints(IPlatformGenModel.class)) {
                removePackageFromRegistry(genModel);
            }
        }

    };

    private EPackageHelper() {
        // Prevent instantiation
    }

    public static EPackageHelper getInstance() {
        return INSTANCE;
    }

    public void dispose() {
        EGFPlatformPlugin.getPlatformManager().removePlatformExtensionPointListener(_extensionPointListener);
    }

    private void init() {
        if (nsuri2basePackage != null) {
            return;
        }
        nsuri2basePackage = new HashMap<String, String>();
        for (IPlatformGenModel genModel : EGFCorePlugin.getPlatformGenModels()) {
            addEcoreModel(genModel);
        }
        EGFPlatformPlugin.getPlatformManager().addPlatformExtensionPointListener(_extensionPointListener);
    }

    private void addEcoreModel(IPlatformGenModel genModel) {
        try {
            try {
                URI uri = genModel.getGenModelURI();
                if (uri == null)
                    handleClassname(genModel);
                else
                    handleURI(uri);
            } catch (Exception e) {
                // don't care since we will try another way.
            }
            addPackage2registry(genModel);
        } catch (Exception e) {
            EGFPatternPlugin.getDefault().logError(e);
        }

    }

    private void handleClassname(IPlatformGenModel genModel) {
        String classname = genModel.getGeneratedPackage();
        int index = classname.lastIndexOf("."); //$NON-NLS-1$
        if (index == -1) {
            throw new IllegalStateException();
        }
        if (index == 0) {
            nsuri2basePackage.put(genModel.getNamespace(), ""); //$NON-NLS-1$
        } else {
            // to remove the last dot
            nsuri2basePackage.put(genModel.getNamespace(), classname.substring(0, index));
        }

    }

    private void handleURI(URI uri) {
        ResourceSet set = new EGFResourceSet();
        Resource res = set.getResource(uri, true);
        try {
            for (EObject obj : res.getContents()) {
                if (obj instanceof GenModel) {
                    GenModel genModel2 = (GenModel) obj;
                    for (GenPackage gPack : genModel2.getAllGenPackagesWithClassifiers()) {
                        EPackage ecorePackage = gPack.getEcorePackage();
                        String basePackageName = gPack.getInterfacePackageName();
                        nsuri2basePackage.put(ecorePackage.getNsURI(), basePackageName);
                    }
                }
            }
        } finally {
            res.unload();
        }
    }

    public String getBasePackage(EPackage ePackage) {
        if (nsuri2basePackage == null) {
            init();
        }
        String nsURI = ePackage.getNsURI();
        String name = nsuri2basePackage.get(nsURI);
        if (name == null)
            throw new IllegalStateException();
        return name;
    }

    private void addPackage2registry(IPlatformGenModel genModel) {
        String nsURI = genModel.getNamespace();
        if (!REGISTRY.containsKey(nsURI))
            REGISTRY.put(nsURI, new Descriptor2(genModel));
    }

    private void removePackageFromRegistry(IPlatformGenModel genModel) {
        String nsURI = genModel.getNamespace();
        REGISTRY.remove(nsURI);
    }

    public EPackage getEPackage(EObject eObject) {
        if (eObject == null) {
            return null;
        }
        if (eObject instanceof EPackage) {
            return (EPackage) eObject;
        } else if (eObject instanceof EClassifier) {
            return ((EClassifier) eObject).getEPackage();
        } else if (eObject instanceof EOperation) {
            return ((EOperation) eObject).getEContainingClass().getEPackage();
        } else if (eObject instanceof EStructuralFeature) {
            return ((EStructuralFeature) eObject).getEContainingClass().getEPackage();
        } else if (eObject instanceof EAnnotation) {
            return getEPackage(((EAnnotation) eObject).getEModelElement());
        } else if (eObject instanceof EParameter) {
            return getEPackage(((EParameter) eObject).getEOperation());
        }
        throw new UnsupportedOperationException(NLS.bind("EPackage couldn't be resolved ''{0}''", EcoreUtil.getURI(eObject))); //$NON-NLS-1$
    }

    public static class RegistrationException extends Exception {

        private static final long serialVersionUID = 1L;

        private RegistrationException(String message, Throwable cause) {
            super(message, cause);
        }

    }

    private static class Descriptor2 implements EPackage.Descriptor {

        private final IPlatformGenModel genModel;

        private Descriptor2(IPlatformGenModel genModel) {
            super();
            this.genModel = genModel;

        }

        public EFactory getEFactory() {

            return null;
        }

        public EPackage getEPackage() {
            try {
                Class<?> javaClass = getGenModel().getPlatformBundle().getBundle().loadClass(getGenModel().getGeneratedPackage());
                Field field = javaClass.getField(INSTANCE_FIELD_NAME);
                Object result = field.get(null);
                return (EPackage) result;
            } catch (ClassNotFoundException e) {
                throw new WrappedException(e);
            } catch (IllegalAccessException e) {
                throw new WrappedException(e);
            } catch (NoSuchFieldException e) {
                throw new WrappedException(e);
            }
        }

        public IPlatformGenModel getGenModel() {
            return genModel;
        }
    }
}
