## ----include=FALSE------------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup, message=FALSE, results="hide"-------------------------------------
library(shiny)
library(shiny.gosling)

## ----reading-data-------------------------------------------------------------

track1_data <- track_data(
  url = "https://server.gosling-lang.org/api/v1/tileset_info/?d=NC_045512_2-multivec",
  type = "multivec",
  row = "base",
  column = "position",
  value = "count",
  categories = c("A", "T", "G", "C"),
  start = "start",
  end = "end"
)

## ----single_track-------------------------------------------------------------

track1 <- add_single_track(
  mark = "bar",
  y = visual_channel_y(
    field = "count", type = "quantitative", axis = "none"
  )
)

track2 <- add_single_track(
  dataTransform = track_data_transform(
    type = "filter",
    field = "count",
    oneOf = list(0),
    not = TRUE
  ),
  mark = "text",
  x = visual_channel_x(
    field = "start", type = "genomic"
  ),
  xe = visual_channel_x(
    field = "end", type = "genomic"
  ),
  size = 24,
  color = "white",
  visibility = list(list(
    operation = "less-than",
    measure = "width",
    threshold = "|xe-x|",
    transitionPadding = 30,
    target = "mark"
  ),
  list(
    operation = "LT",
    measure = "zoomLevel",
    threshold = 40,
    target = "track"
  ))
)

## ----visual_channels----------------------------------------------------------

track1_x <- visual_channel_x(
  field = "position", type = "genomic"
)

track1_color <- visual_channel_color(
  field = "base",
  type = "nominal",
  domain = c("A", "T", "G", "C"),
  legend = TRUE
)

track1_text <- visual_channel_text(
  field = "base", type = "nominal"
)

track1_style <- default_track_styles(
  inlineLegend = TRUE
)

## ----combined_track-----------------------------------------------------------

track3 <- add_single_track(
  title = "NC_045512.2 Sequence",
  alignment = "overlay",
  data = track1_data,
  tracks = add_multi_tracks(
    track1, track2
  ),
  x = track1_x,
  color = track1_color,
  text = track1_text,
  style = track1_style,
  width = 800, height = 40
)

## ----compose------------------------------------------------------------------

view1 <- compose_view(
  multi = TRUE,
  centerRadius = 0,
  xDomain = list(interval = c(1, 29903)),
  linkingId = "detail",
  alignment = "stack",
  tracks = add_multi_tracks(
    track3
  )
)

## ----arrange------------------------------------------------------------------

combined_view <- arrange_views(
  title = "SARS-CoV-2",
  subtitle = "Data Source: WashU Virus Genome Browser, NCBI, GISAID",
  assembly = list(list("NC_045512.2", 29903)),
  layout = "linear",
  spacing = 50,
  views = list(view1),
  listify = FALSE
)

## ----ui, results="hide"-------------------------------------------------------

ui <- fluidPage(
  use_gosling(),
  fluidRow(
    column(6, goslingOutput("gosling_plot"))
  )
)


server <- function(input, output, session) {
  output$gosling_plot <- renderGosling({
    gosling(
      component_id = "sars_cov2",
      combined_view
    )
  })
}

shinyApp(ui, server)

## ----session_info-------------------------------------------------------------

sessionInfo()

