## ----setup, include=FALSE-----------------------------------------------------
options("datatable.print.nrows" = 50)

## ----install, eval=FALSE------------------------------------------------------
# if (!require("BiocManager"))
#     install.packages("BiocManager")
# BiocManager::install("iscream")

## ----load_iscream-------------------------------------------------------------
library(iscream)
set_threads(2)

## ----load_data----------------------------------------------------------------
data_dir <- system.file("extdata", package = "iscream")
(bedfiles <- list.files(
  data_dir,
  pattern = "cell[1-5].bed.gz$",
  full.names = TRUE))

## ----make_regions-------------------------------------------------------------
regions <- c("chr1:184577-680065", "chrY:56877780-56882524")

## ----query_chroms-------------------------------------------------------------
query_chroms(bedfiles)

## ----tabix_one----------------------------------------------------------------
tabix(bedfiles[1], regions)

## ----tabix_all----------------------------------------------------------------
tabix(bedfiles, regions)

## ----tabix_gr, message=FALSE--------------------------------------------------
if (!require("GenomicRanges", quietly = TRUE)) {
  stop("The 'GenomicRanges' package must be installed for this functionality")
}
gr <- GRanges(regions)
tabix_gr(bedfiles, gr)

## ----tabix_colnames-----------------------------------------------------------
tabix_gr(bedfiles, regions, col.names = c("beta", "coverage"))

## ----tabix_raw----------------------------------------------------------------
tabix_raw(bedfiles, regions)

## ----tabix_wgbs---------------------------------------------------------------
tabix_gr(bedfiles, regions, aligner = "biscuit")

## ----summary------------------------------------------------------------------
summarize_regions(
  bedfiles,
  regions,
  columns = c(4, 5),
  col_names = c("beta", "coverage")
)

## ----summary_named------------------------------------------------------------
names(regions) <- c("R1", "R2")
summarize_regions(
  bedfiles,
  regions,
  fun = c("mean", "sum"),
  columns = 5,
  col_names = "coverage"
)

## ----summarize_meth_regions---------------------------------------------------
summarize_meth_regions(
  bedfiles,
  regions,
  aligner = "biscuit",
  fun = c("mean", "sum")
)

## ----make_mat, message=FALSE--------------------------------------------------
if (!require("SummarizedExperiment", quietly = TRUE)) {
  stop("The 'SummarizedExperiment' package must be installed for this functionality")
}
(mat <- make_mat_se(bedfiles, regions, column = 4, mat_name = "beta"))
head(assay(mat), 10)

## ----sparse-------------------------------------------------------------------
mat <- make_mat(bedfiles, regions, column = 4, mat_name = "beta", sparse = TRUE)
head(mat$beta, 10)

## ----message=FALSE------------------------------------------------------------
if (require("bsseq", quietly = TRUE)) {
  meth_mat <- make_mat_bsseq(bedfiles, regions)
  do.call(BSseq, meth_mat)
}

## ----si-----------------------------------------------------------------------
sessionInfo()

