"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReduxStoreAdapter = exports.BehaviorSubjectAdapter = void 0;
exports.isStateAdapter = isStateAdapter;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/* eslint-disable max-classes-per-file */

/**
 * @experimental
 * StateAdapter provides a common interface for various state management backends.
 * Implementations can wrap Redux Store slices, BehaviorSubjects, or other state containers.
 */

/**
 * @experimental
 * Adapter implementation for BehaviorSubject-based state
 */
class BehaviorSubjectAdapter {
  constructor(state$) {
    this.state$ = state$;
  }
  getState() {
    return this.state$.getValue();
  }
  setState(updater) {
    const current = this.state$.getValue();
    const updated = updater(current);
    this.state$.next(updated);
  }
  subscribe(listener) {
    const subscription = this.state$.subscribe(listener);
    return () => subscription.unsubscribe();
  }
}

/**
 * @experimental
 * Adapter implementation for Redux store slices
 */
exports.BehaviorSubjectAdapter = BehaviorSubjectAdapter;
class ReduxStoreAdapter {
  constructor(store, stateSelector, createAction) {
    this.store = store;
    this.stateSelector = stateSelector;
    this.createAction = createAction;
  }
  getState() {
    return this.stateSelector(this.store.getState());
  }
  setState(updater) {
    this.store.dispatch(this.createAction(updater));
  }
  subscribe(listener) {
    let currentState = this.getState();
    return this.store.subscribe(() => {
      const nextState = this.getState();
      // Only call listener if the selected state has changed
      if (nextState !== currentState) {
        currentState = nextState;
        listener(nextState);
      }
    });
  }
}

/**
 * @experimental
 * Type guard to check if an object is a StateAdapter
 */
exports.ReduxStoreAdapter = ReduxStoreAdapter;
function isStateAdapter(obj) {
  return obj && typeof obj.getState === 'function' && typeof obj.setState === 'function' && typeof obj.subscribe === 'function';
}