// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import LRU from 'lru-cache';
import { needs, isEncryptionMaterial, isDecryptionMaterial, isBranchKeyMaterial, } from '@aws-crypto/material-management';
export function getLocalCryptographicMaterialsCache(capacity, proactiveFrequency = 1000 * 60) {
    const cache = new LRU({
        max: capacity,
        dispose(_key, value) {
            /* Zero out the unencrypted dataKey, when the material is removed from the cache. */
            value.response.zeroUnencryptedDataKey();
        },
    });
    (function proactivelyTryAndEvictTail() {
        const timeout = setTimeout(() => {
            mayEvictTail();
            proactivelyTryAndEvictTail();
        }, proactiveFrequency);
        /* In Node.js the event loop will _only_ exit if there are no outstanding events.
         * This means that if I did nothing the event loop would *always* be blocked.
         * This is unfortunate and very bad for things like Lambda.
         * So, I tell Node.js to not wait for this timer.
         * See: https://nodejs.org/api/timers.html#timers_timeout_unref
         */
        // @ts-ignore
        timeout.unref && timeout.unref();
    })();
    return {
        putEncryptionMaterial(key, material, plaintextLength, maxAge) {
            /* Precondition: putEncryptionMaterial plaintextLength can not be negative. */
            needs(plaintextLength >= 0, 'Malformed plaintextLength');
            /* Precondition: Only cache EncryptionMaterial. */
            needs(isEncryptionMaterial(material), 'Malformed response.');
            /* Precondition: Only cache EncryptionMaterial that is cacheSafe. */
            needs(material.suite.cacheSafe, 'Can not cache non-cache safe material');
            const entry = Object.seal({
                response: material,
                bytesEncrypted: plaintextLength,
                messagesEncrypted: 1,
                now: Date.now(),
            });
            cache.set(key, entry, maxAge);
        },
        putDecryptionMaterial(key, material, maxAge) {
            /* Precondition: Only cache DecryptionMaterial. */
            needs(isDecryptionMaterial(material), 'Malformed response.');
            /* Precondition: Only cache DecryptionMaterial that is cacheSafe. */
            needs(material.suite.cacheSafe, 'Can not cache non-cache safe material');
            const entry = Object.seal({
                response: material,
                bytesEncrypted: 0,
                messagesEncrypted: 0,
                now: Date.now(),
            });
            cache.set(key, entry, maxAge);
        },
        putBranchKeyMaterial(key, material, maxAge) {
            /* Precondition: Only cache BranchKeyMaterial */
            needs(isBranchKeyMaterial(material), 'Malformed response.');
            const entry = Object.seal({
                response: material,
                now: Date.now(),
            });
            cache.set(key, entry, maxAge);
        },
        getEncryptionMaterial(key, plaintextLength) {
            /* Precondition: plaintextLength can not be negative. */
            needs(plaintextLength >= 0, 'Malformed plaintextLength');
            const entry = cache.get(key);
            /* Check for early return (Postcondition): If this key does not have an EncryptionMaterial, return false. */
            if (!entry)
                return false;
            /* Postcondition: Only return EncryptionMaterial. */
            needs(isEncryptionMaterial(entry.response), 'Malformed response.');
            const encryptionMaterialEntry = entry;
            encryptionMaterialEntry.bytesEncrypted += plaintextLength;
            encryptionMaterialEntry.messagesEncrypted += 1;
            return entry;
        },
        getDecryptionMaterial(key) {
            const entry = cache.get(key);
            /* Check for early return (Postcondition): If this key does not have a DecryptionMaterial, return false. */
            if (!entry)
                return false;
            /* Postcondition: Only return DecryptionMaterial. */
            needs(isDecryptionMaterial(entry.response), 'Malformed response.');
            return entry;
        },
        getBranchKeyMaterial(key) {
            const entry = cache.get(key);
            /* Postcondition: If this key does not have a BranchKeyMaterial, return false */
            if (!entry)
                return false;
            /* Postcondition: Only return BranchKeyMaterial */
            needs(isBranchKeyMaterial(entry.response), 'Malformed response.');
            return entry;
        },
        del(key) {
            cache.del(key);
        },
    };
    function mayEvictTail() {
        // @ts-ignore
        const { tail } = cache.dumpLru();
        /* Check for early return (Postcondition) UNTESTED: If there is no tail, then the cache is empty. */
        if (!tail)
            return;
        /* The underlying Yallist tail Node has a `value`.
         * This value is a lru-cache Entry and has a `key`.
         */
        const { key } = tail.value;
        // Peek will evict, but not update the "recently used"-ness of the key.
        cache.peek(key);
    }
}
//# sourceMappingURL=data:application/json;base64,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