﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 * <p>Represents the request to create the user import job.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-idp-2016-04-18/CreateUserImportJobRequest">AWS
 * API Reference</a></p>
 */
class CreateUserImportJobRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API CreateUserImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateUserImportJob"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A friendly name for the user import job.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  CreateUserImportJobRequest& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the user pool that you want to import users into.</p>
   */
  inline const Aws::String& GetUserPoolId() const { return m_userPoolId; }
  inline bool UserPoolIdHasBeenSet() const { return m_userPoolIdHasBeenSet; }
  template <typename UserPoolIdT = Aws::String>
  void SetUserPoolId(UserPoolIdT&& value) {
    m_userPoolIdHasBeenSet = true;
    m_userPoolId = std::forward<UserPoolIdT>(value);
  }
  template <typename UserPoolIdT = Aws::String>
  CreateUserImportJobRequest& WithUserPoolId(UserPoolIdT&& value) {
    SetUserPoolId(std::forward<UserPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>You must specify an IAM role that has permission to log import-job results to
   * Amazon CloudWatch Logs. This parameter is the ARN of that role.</p>
   */
  inline const Aws::String& GetCloudWatchLogsRoleArn() const { return m_cloudWatchLogsRoleArn; }
  inline bool CloudWatchLogsRoleArnHasBeenSet() const { return m_cloudWatchLogsRoleArnHasBeenSet; }
  template <typename CloudWatchLogsRoleArnT = Aws::String>
  void SetCloudWatchLogsRoleArn(CloudWatchLogsRoleArnT&& value) {
    m_cloudWatchLogsRoleArnHasBeenSet = true;
    m_cloudWatchLogsRoleArn = std::forward<CloudWatchLogsRoleArnT>(value);
  }
  template <typename CloudWatchLogsRoleArnT = Aws::String>
  CreateUserImportJobRequest& WithCloudWatchLogsRoleArn(CloudWatchLogsRoleArnT&& value) {
    SetCloudWatchLogsRoleArn(std::forward<CloudWatchLogsRoleArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobName;

  Aws::String m_userPoolId;

  Aws::String m_cloudWatchLogsRoleArn;
  bool m_jobNameHasBeenSet = false;
  bool m_userPoolIdHasBeenSet = false;
  bool m_cloudWatchLogsRoleArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
