﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/waf-regional/WAFRegionalRequest.h>
#include <aws/waf-regional/WAFRegional_EXPORTS.h>
#include <aws/waf-regional/model/RegexPatternSetUpdate.h>

#include <utility>

namespace Aws {
namespace WAFRegional {
namespace Model {

/**
 */
class UpdateRegexPatternSetRequest : public WAFRegionalRequest {
 public:
  AWS_WAFREGIONAL_API UpdateRegexPatternSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRegexPatternSet"; }

  AWS_WAFREGIONAL_API Aws::String SerializePayload() const override;

  AWS_WAFREGIONAL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>RegexPatternSetId</code> of the <a>RegexPatternSet</a> that you
   * want to update. <code>RegexPatternSetId</code> is returned by
   * <a>CreateRegexPatternSet</a> and by <a>ListRegexPatternSets</a>.</p>
   */
  inline const Aws::String& GetRegexPatternSetId() const { return m_regexPatternSetId; }
  inline bool RegexPatternSetIdHasBeenSet() const { return m_regexPatternSetIdHasBeenSet; }
  template <typename RegexPatternSetIdT = Aws::String>
  void SetRegexPatternSetId(RegexPatternSetIdT&& value) {
    m_regexPatternSetIdHasBeenSet = true;
    m_regexPatternSetId = std::forward<RegexPatternSetIdT>(value);
  }
  template <typename RegexPatternSetIdT = Aws::String>
  UpdateRegexPatternSetRequest& WithRegexPatternSetId(RegexPatternSetIdT&& value) {
    SetRegexPatternSetId(std::forward<RegexPatternSetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of <code>RegexPatternSetUpdate</code> objects that you want to
   * insert into or delete from a <a>RegexPatternSet</a>.</p>
   */
  inline const Aws::Vector<RegexPatternSetUpdate>& GetUpdates() const { return m_updates; }
  inline bool UpdatesHasBeenSet() const { return m_updatesHasBeenSet; }
  template <typename UpdatesT = Aws::Vector<RegexPatternSetUpdate>>
  void SetUpdates(UpdatesT&& value) {
    m_updatesHasBeenSet = true;
    m_updates = std::forward<UpdatesT>(value);
  }
  template <typename UpdatesT = Aws::Vector<RegexPatternSetUpdate>>
  UpdateRegexPatternSetRequest& WithUpdates(UpdatesT&& value) {
    SetUpdates(std::forward<UpdatesT>(value));
    return *this;
  }
  template <typename UpdatesT = RegexPatternSetUpdate>
  UpdateRegexPatternSetRequest& AddUpdates(UpdatesT&& value) {
    m_updatesHasBeenSet = true;
    m_updates.emplace_back(std::forward<UpdatesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value returned by the most recent call to <a>GetChangeToken</a>.</p>
   */
  inline const Aws::String& GetChangeToken() const { return m_changeToken; }
  inline bool ChangeTokenHasBeenSet() const { return m_changeTokenHasBeenSet; }
  template <typename ChangeTokenT = Aws::String>
  void SetChangeToken(ChangeTokenT&& value) {
    m_changeTokenHasBeenSet = true;
    m_changeToken = std::forward<ChangeTokenT>(value);
  }
  template <typename ChangeTokenT = Aws::String>
  UpdateRegexPatternSetRequest& WithChangeToken(ChangeTokenT&& value) {
    SetChangeToken(std::forward<ChangeTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_regexPatternSetId;
  bool m_regexPatternSetIdHasBeenSet = false;

  Aws::Vector<RegexPatternSetUpdate> m_updates;
  bool m_updatesHasBeenSet = false;

  Aws::String m_changeToken;
  bool m_changeTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAFRegional
}  // namespace Aws
