/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Scheduler;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import io.opentelemetry.api.common.Attributes;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.rocketmq.common.BoundaryType;
import org.apache.rocketmq.common.message.MessageQueue;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.store.GetMessageResult;
import org.apache.rocketmq.store.GetMessageStatus;
import org.apache.rocketmq.store.MessageFilter;
import org.apache.rocketmq.store.QueryMessageResult;
import org.apache.rocketmq.store.SelectMappedBufferResult;
import org.apache.rocketmq.tieredstore.MessageStoreFetcher;
import org.apache.rocketmq.tieredstore.common.GetMessageResultExt;
import org.apache.rocketmq.tieredstore.common.InFlightRequestFuture;
import org.apache.rocketmq.tieredstore.common.MessageCacheKey;
import org.apache.rocketmq.tieredstore.common.SelectBufferResult;
import org.apache.rocketmq.tieredstore.common.SelectBufferResultWrapper;
import org.apache.rocketmq.tieredstore.common.TieredMessageStoreConfig;
import org.apache.rocketmq.tieredstore.common.TieredStoreExecutor;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.file.CompositeFlatFile;
import org.apache.rocketmq.tieredstore.file.CompositeQueueFlatFile;
import org.apache.rocketmq.tieredstore.file.TieredFlatFileManager;
import org.apache.rocketmq.tieredstore.index.IndexItem;
import org.apache.rocketmq.tieredstore.index.IndexService;
import org.apache.rocketmq.tieredstore.metadata.TieredMetadataStore;
import org.apache.rocketmq.tieredstore.metadata.TopicMetadata;
import org.apache.rocketmq.tieredstore.metrics.TieredStoreMetricsManager;
import org.apache.rocketmq.tieredstore.util.CQItemBufferUtil;
import org.apache.rocketmq.tieredstore.util.MessageBufferUtil;
import org.apache.rocketmq.tieredstore.util.TieredStoreUtil;

public class TieredMessageFetcher
implements MessageStoreFetcher {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    private final String brokerName;
    private final TieredMetadataStore metadataStore;
    private final TieredMessageStoreConfig storeConfig;
    private final TieredFlatFileManager flatFileManager;
    private final Cache<MessageCacheKey, SelectBufferResultWrapper> readAheadCache;

    public TieredMessageFetcher(TieredMessageStoreConfig storeConfig) {
        this.storeConfig = storeConfig;
        this.brokerName = storeConfig.getBrokerName();
        this.metadataStore = TieredStoreUtil.getMetadataStore(storeConfig);
        this.flatFileManager = TieredFlatFileManager.getInstance(storeConfig);
        this.readAheadCache = this.initCache(storeConfig);
    }

    private Cache<MessageCacheKey, SelectBufferResultWrapper> initCache(TieredMessageStoreConfig storeConfig) {
        long memoryMaxSize = (long)((double)Runtime.getRuntime().maxMemory() * storeConfig.getReadAheadCacheSizeThresholdRate());
        return Caffeine.newBuilder().scheduler(Scheduler.systemScheduler()).expireAfterWrite(storeConfig.getReadAheadCacheExpireDuration(), TimeUnit.MILLISECONDS).maximumWeight(memoryMaxSize).weigher((key, msg) -> msg.getBufferSize()).recordStats().build();
    }

    @VisibleForTesting
    public Cache<MessageCacheKey, SelectBufferResultWrapper> getMessageCache() {
        return this.readAheadCache;
    }

    protected void putMessageToCache(CompositeFlatFile flatFile, SelectBufferResultWrapper result) {
        this.readAheadCache.put((Object)new MessageCacheKey(flatFile, result.getOffset()), (Object)result);
    }

    protected SelectBufferResultWrapper getMessageFromCache(CompositeFlatFile flatFile, long offset) {
        return (SelectBufferResultWrapper)this.readAheadCache.getIfPresent((Object)new MessageCacheKey(flatFile, offset));
    }

    protected void recordCacheAccess(CompositeFlatFile flatFile, String group, long offset, List<SelectBufferResultWrapper> resultWrapperList) {
        if (!resultWrapperList.isEmpty()) {
            offset = resultWrapperList.get(resultWrapperList.size() - 1).getOffset();
        }
        flatFile.recordGroupAccess(group, offset);
        resultWrapperList.forEach(wrapper -> {
            if ((long)wrapper.incrementAndGet() >= flatFile.getActiveGroupCount()) {
                this.readAheadCache.invalidate((Object)new MessageCacheKey(flatFile, wrapper.getOffset()));
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void prefetchMessage(CompositeQueueFlatFile flatFile, String group, int maxCount, long nextBeginOffset) {
        if (maxCount == 1 || flatFile.getReadAheadFactor() == 1) {
            return;
        }
        int prefetchBatchSize = Math.min(maxCount * flatFile.getReadAheadFactor(), this.storeConfig.getReadAheadMessageCountThreshold());
        InFlightRequestFuture inflightRequest = flatFile.getInflightRequest(group, nextBeginOffset, prefetchBatchSize);
        if (!inflightRequest.isAllDone()) {
            return;
        }
        CompositeQueueFlatFile compositeQueueFlatFile = flatFile;
        synchronized (compositeQueueFlatFile) {
            boolean lastRequestIsExpired;
            inflightRequest = flatFile.getInflightRequest(nextBeginOffset, maxCount);
            if (!inflightRequest.isAllDone()) {
                return;
            }
            long maxOffsetOfLastRequest = inflightRequest.getLastFuture().join();
            boolean bl = lastRequestIsExpired = this.getMessageFromCache(flatFile, nextBeginOffset) == null;
            if (lastRequestIsExpired || maxOffsetOfLastRequest != -1L && nextBeginOffset >= inflightRequest.getStartOffset()) {
                long queueOffset;
                if (lastRequestIsExpired) {
                    queueOffset = nextBeginOffset;
                    flatFile.decreaseReadAheadFactor();
                } else {
                    queueOffset = maxOffsetOfLastRequest + 1L;
                    flatFile.increaseReadAheadFactor();
                }
                int factor = Math.min(flatFile.getReadAheadFactor(), this.storeConfig.getReadAheadMessageCountThreshold() / maxCount);
                int flag = 0;
                int concurrency = 1;
                if (factor > this.storeConfig.getReadAheadBatchSizeFactorThreshold()) {
                    flag = factor % this.storeConfig.getReadAheadBatchSizeFactorThreshold() == 0 ? 0 : 1;
                    concurrency = factor / this.storeConfig.getReadAheadBatchSizeFactorThreshold() + flag;
                }
                int requestBatchSize = maxCount * Math.min(factor, this.storeConfig.getReadAheadBatchSizeFactorThreshold());
                ArrayList<Pair<Integer, CompletableFuture<Long>>> futureList = new ArrayList<Pair<Integer, CompletableFuture<Long>>>();
                long nextQueueOffset = queueOffset;
                if (flag == 1) {
                    int firstBatchSize = factor % this.storeConfig.getReadAheadBatchSizeFactorThreshold() * maxCount;
                    CompletableFuture<Long> future = this.prefetchMessageThenPutToCache(flatFile, nextQueueOffset, firstBatchSize);
                    futureList.add((Pair<Integer, CompletableFuture<Long>>)Pair.of((Object)firstBatchSize, future));
                    nextQueueOffset += (long)firstBatchSize;
                }
                for (long i = 0L; i < (long)(concurrency - flag); ++i) {
                    CompletableFuture<Long> future = this.prefetchMessageThenPutToCache(flatFile, nextQueueOffset + i * (long)requestBatchSize, requestBatchSize);
                    futureList.add((Pair<Integer, CompletableFuture<Long>>)Pair.of((Object)requestBatchSize, future));
                }
                flatFile.putInflightRequest(group, queueOffset, maxCount * factor, futureList);
                LOGGER.debug("TieredMessageFetcher#preFetchMessage: try to prefetch messages for later requests: next begin offset: {}, request offset: {}, factor: {}, flag: {}, request batch: {}, concurrency: {}", new Object[]{nextBeginOffset, queueOffset, factor, flag, requestBatchSize, concurrency});
            }
        }
    }

    private CompletableFuture<Long> prefetchMessageThenPutToCache(CompositeQueueFlatFile flatFile, long queueOffset, int batchSize) {
        MessageQueue mq = flatFile.getMessageQueue();
        return this.getMessageFromTieredStoreAsync(flatFile, queueOffset, batchSize).thenApply(result -> {
            if (result.getStatus() == GetMessageStatus.OFFSET_OVERFLOW_ONE || result.getStatus() == GetMessageStatus.OFFSET_OVERFLOW_BADLY) {
                return -1L;
            }
            if (result.getStatus() != GetMessageStatus.FOUND) {
                LOGGER.warn("MessageFetcher prefetch message then put to cache failed, result: {}, topic: {}, queue: {}, queue offset: {}, batch size: {}", new Object[]{result.getStatus(), mq.getTopic(), mq.getQueueId(), queueOffset, batchSize});
                return -1L;
            }
            try {
                List offsetList = result.getMessageQueueOffset();
                List<Long> tagCodeList = result.getTagCodeList();
                List msgList = result.getMessageMapedList();
                for (int i = 0; i < offsetList.size(); ++i) {
                    SelectMappedBufferResult msg = (SelectMappedBufferResult)msgList.get(i);
                    SelectBufferResultWrapper bufferResult = new SelectBufferResultWrapper(msg, (Long)offsetList.get(i), tagCodeList.get(i), false);
                    this.putMessageToCache(flatFile, bufferResult);
                }
                return (Long)offsetList.get(offsetList.size() - 1);
            }
            catch (Exception e) {
                LOGGER.error("MessageFetcher prefetch message then put to cache failed, topic: {}, queue: {}, queue offset: {}, batch size: {}", new Object[]{mq.getTopic(), mq.getQueueId(), queueOffset, batchSize, e});
                return -1L;
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<GetMessageResultExt> getMessageFromCacheAsync(CompositeQueueFlatFile flatFile, String group, long queueOffset, int maxCount, boolean waitInflightRequest) {
        CompletionStage resultFuture;
        CompletableFuture<Long> future;
        SelectBufferResultWrapper wrapper2;
        int i;
        MessageQueue mq = flatFile.getMessageQueue();
        long lastGetOffset = queueOffset - 1L;
        ArrayList<SelectBufferResultWrapper> resultWrapperList = new ArrayList<SelectBufferResultWrapper>(maxCount);
        for (i = 0; i < maxCount; ++i) {
            if ((wrapper2 = this.getMessageFromCache(flatFile, ++lastGetOffset)) == null) {
                --lastGetOffset;
                break;
            }
            resultWrapperList.add(wrapper2);
        }
        if (waitInflightRequest) {
            Attributes attributes = TieredStoreMetricsManager.newAttributesBuilder().put("topic", mq.getTopic()).put("group", group).build();
            TieredStoreMetricsManager.cacheAccess.add((long)maxCount, attributes);
            TieredStoreMetricsManager.cacheHit.add((long)resultWrapperList.size(), attributes);
        }
        if (resultWrapperList.isEmpty() && waitInflightRequest && !(future = flatFile.getInflightRequest(group, queueOffset, maxCount).getFuture(queueOffset)).isDone()) {
            Stopwatch stopwatch = Stopwatch.createStarted();
            return future.thenComposeAsync(v -> {
                LOGGER.debug("MessageFetcher#getMessageFromCacheAsync: wait for response cost: {}ms", (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
                return this.getMessageFromCacheAsync(flatFile, group, queueOffset, maxCount, false);
            }, (Executor)TieredStoreExecutor.fetchDataExecutor);
        }
        for (i = 0; i < maxCount - resultWrapperList.size(); ++i) {
            if ((wrapper2 = this.getMessageFromCache(flatFile, ++lastGetOffset)) == null) {
                --lastGetOffset;
                break;
            }
            resultWrapperList.add(wrapper2);
        }
        this.recordCacheAccess(flatFile, group, queueOffset, resultWrapperList);
        if (!resultWrapperList.isEmpty()) {
            LOGGER.debug("MessageFetcher cache hit, group: {}, topic: {}, queueId: {}, offset: {}, maxCount: {}, resultSize: {}", new Object[]{group, mq.getTopic(), mq.getQueueId(), queueOffset, maxCount, resultWrapperList.size()});
            GetMessageResultExt result2 = new GetMessageResultExt();
            result2.setStatus(GetMessageStatus.FOUND);
            result2.setMinOffset(flatFile.getConsumeQueueMinOffset());
            result2.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
            result2.setNextBeginOffset(queueOffset + (long)resultWrapperList.size());
            resultWrapperList.forEach(wrapper -> result2.addMessageExt(wrapper.getDuplicateResult(), wrapper.getOffset(), wrapper.getTagCode()));
            if (lastGetOffset < result2.getMaxOffset()) {
                this.prefetchMessage(flatFile, group, maxCount, lastGetOffset + 1L);
            }
            return CompletableFuture.completedFuture(result2);
        }
        LOGGER.info("MessageFetcher cache miss, group: {}, topic: {}, queueId: {}, offset: {}, maxCount: {}", new Object[]{group, mq.getTopic(), mq.getQueueId(), queueOffset, maxCount});
        CompositeQueueFlatFile compositeQueueFlatFile = flatFile;
        synchronized (compositeQueueFlatFile) {
            int batchSize = maxCount * this.storeConfig.getReadAheadMinFactor();
            resultFuture = this.getMessageFromTieredStoreAsync(flatFile, queueOffset, batchSize).thenApply(result -> {
                if (result.getStatus() != GetMessageStatus.FOUND) {
                    return result;
                }
                GetMessageResultExt newResult = new GetMessageResultExt();
                List offsetList = result.getMessageQueueOffset();
                List<Long> tagCodeList = result.getTagCodeList();
                List msgList = result.getMessageMapedList();
                for (int i = 0; i < offsetList.size(); ++i) {
                    SelectMappedBufferResult msg = (SelectMappedBufferResult)msgList.get(i);
                    SelectBufferResultWrapper bufferResult = new SelectBufferResultWrapper(msg, (Long)offsetList.get(i), tagCodeList.get(i), true);
                    this.putMessageToCache(flatFile, bufferResult);
                    if (newResult.getMessageMapedList().size() >= maxCount) continue;
                    newResult.addMessageExt(msg, (Long)offsetList.get(i), tagCodeList.get(i));
                }
                newResult.setStatus(GetMessageStatus.FOUND);
                newResult.setMinOffset(flatFile.getConsumeQueueMinOffset());
                newResult.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
                newResult.setNextBeginOffset(queueOffset + (long)newResult.getMessageMapedList().size());
                return newResult;
            });
            ArrayList<Pair<Integer, CompletableFuture<Long>>> futureList = new ArrayList<Pair<Integer, CompletableFuture<Long>>>();
            CompletionStage inflightRequestFuture = ((CompletableFuture)resultFuture).thenApply(result -> result.getStatus() == GetMessageStatus.FOUND ? (Long)result.getMessageQueueOffset().get(result.getMessageQueueOffset().size() - 1) : Long.valueOf(-1L));
            futureList.add(Pair.of((Object)batchSize, (Object)inflightRequestFuture));
            flatFile.putInflightRequest(group, queueOffset, batchSize, futureList);
        }
        return resultFuture;
    }

    public CompletableFuture<GetMessageResultExt> getMessageFromTieredStoreAsync(CompositeQueueFlatFile flatFile, long queueOffset, int batchSize) {
        CompletableFuture<ByteBuffer> readConsumeQueueFuture;
        GetMessageResultExt result = new GetMessageResultExt();
        result.setMinOffset(flatFile.getConsumeQueueMinOffset());
        result.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
        if (queueOffset < result.getMaxOffset()) {
            batchSize = Math.min(batchSize, (int)Math.min(result.getMaxOffset() - queueOffset, Integer.MAX_VALUE));
        } else {
            if (queueOffset == result.getMaxOffset()) {
                result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_ONE);
                result.setNextBeginOffset(queueOffset);
                return CompletableFuture.completedFuture(result);
            }
            if (queueOffset > result.getMaxOffset()) {
                result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_BADLY);
                result.setNextBeginOffset(result.getMaxOffset());
                return CompletableFuture.completedFuture(result);
            }
        }
        LOGGER.info("MessageFetcher#getMessageFromTieredStoreAsync, topic: {}, queueId: {}, broker offset: {}-{}, offset: {}, expect: {}", new Object[]{flatFile.getMessageQueue().getTopic(), flatFile.getMessageQueue().getQueueId(), result.getMinOffset(), result.getMaxOffset(), queueOffset, batchSize});
        try {
            readConsumeQueueFuture = flatFile.getConsumeQueueAsync(queueOffset, batchSize);
        }
        catch (TieredStoreException e2) {
            switch (e2.getErrorCode()) {
                case NO_NEW_DATA: {
                    result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_ONE);
                    result.setNextBeginOffset(queueOffset);
                    return CompletableFuture.completedFuture(result);
                }
            }
            result.setStatus(GetMessageStatus.OFFSET_FOUND_NULL);
            result.setNextBeginOffset(queueOffset);
            return CompletableFuture.completedFuture(result);
        }
        CompletionStage readCommitLogFuture = readConsumeQueueFuture.thenCompose(cqBuffer -> {
            long firstCommitLogOffset = CQItemBufferUtil.getCommitLogOffset(cqBuffer);
            cqBuffer.position(cqBuffer.remaining() - 20);
            long lastCommitLogOffset = CQItemBufferUtil.getCommitLogOffset(cqBuffer);
            if (lastCommitLogOffset < firstCommitLogOffset) {
                LOGGER.error("MessageFetcher#getMessageFromTieredStoreAsync, last offset is smaller than first offset, topic: {} queueId: {}, offset: {}, firstOffset: {}, lastOffset: {}", new Object[]{flatFile.getMessageQueue().getTopic(), flatFile.getMessageQueue().getQueueId(), queueOffset, firstCommitLogOffset, lastCommitLogOffset});
                return CompletableFuture.completedFuture(ByteBuffer.allocate(0));
            }
            long length = lastCommitLogOffset - firstCommitLogOffset + (long)CQItemBufferUtil.getSize(cqBuffer);
            while (cqBuffer.limit() > 20 && length > (long)this.storeConfig.getReadAheadMessageSizeThreshold()) {
                cqBuffer.limit(cqBuffer.position());
                cqBuffer.position(cqBuffer.limit() - 20);
                length = CQItemBufferUtil.getCommitLogOffset(cqBuffer) - firstCommitLogOffset + (long)CQItemBufferUtil.getSize(cqBuffer);
            }
            return flatFile.getCommitLogAsync(firstCommitLogOffset, (int)length);
        });
        int finalBatchSize = batchSize;
        return ((CompletableFuture)readConsumeQueueFuture.thenCombine(readCommitLogFuture, (cqBuffer, msgBuffer) -> {
            List<SelectBufferResult> bufferList = MessageBufferUtil.splitMessageBuffer(cqBuffer, msgBuffer);
            int requestSize = cqBuffer.remaining() / 20;
            if (bufferList.isEmpty()) {
                result.setStatus(GetMessageStatus.NO_MATCHED_MESSAGE);
                result.setNextBeginOffset(queueOffset + (long)requestSize);
            } else {
                result.setStatus(GetMessageStatus.FOUND);
                result.setNextBeginOffset(queueOffset + (long)requestSize);
                for (SelectBufferResult bufferResult : bufferList) {
                    ByteBuffer slice = bufferResult.getByteBuffer().slice();
                    slice.limit(bufferResult.getSize());
                    SelectMappedBufferResult msg = new SelectMappedBufferResult(bufferResult.getStartOffset(), bufferResult.getByteBuffer(), bufferResult.getSize(), null);
                    result.addMessageExt(msg, MessageBufferUtil.getQueueOffset(slice), bufferResult.getTagCode());
                }
            }
            return result;
        })).exceptionally(e -> {
            MessageQueue mq = flatFile.getMessageQueue();
            LOGGER.warn("MessageFetcher#getMessageFromTieredStoreAsync failed, topic: {} queueId: {}, offset: {}, batchSize: {}", new Object[]{mq.getTopic(), mq.getQueueId(), queueOffset, finalBatchSize, e});
            result.setStatus(GetMessageStatus.OFFSET_FOUND_NULL);
            result.setNextBeginOffset(queueOffset);
            return result;
        });
    }

    @Override
    public CompletableFuture<GetMessageResult> getMessageAsync(String group, String topic, int queueId, long queueOffset, int maxCount, MessageFilter messageFilter) {
        GetMessageResult result = new GetMessageResult();
        CompositeQueueFlatFile flatFile = this.flatFileManager.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            result.setNextBeginOffset(queueOffset);
            result.setStatus(GetMessageStatus.NO_MATCHED_LOGIC_QUEUE);
            return CompletableFuture.completedFuture(result);
        }
        result.setMinOffset(flatFile.getConsumeQueueMinOffset());
        result.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
        if (result.getMaxOffset() <= 0L) {
            result.setStatus(GetMessageStatus.NO_MESSAGE_IN_QUEUE);
            result.setNextBeginOffset(queueOffset);
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset < result.getMinOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_TOO_SMALL);
            result.setNextBeginOffset(result.getMinOffset());
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset == result.getMaxOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_ONE);
            result.setNextBeginOffset(result.getMaxOffset());
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset > result.getMaxOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_BADLY);
            result.setNextBeginOffset(result.getMaxOffset());
            return CompletableFuture.completedFuture(result);
        }
        return this.getMessageFromCacheAsync(flatFile, group, queueOffset, maxCount, true).thenApply(messageResultExt -> messageResultExt.doFilterMessage(messageFilter));
    }

    @Override
    public CompletableFuture<Long> getEarliestMessageTimeAsync(String topic, int queueId) {
        CompositeQueueFlatFile flatFile = this.flatFileManager.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            return CompletableFuture.completedFuture(-1L);
        }
        int length = 64;
        return flatFile.getCommitLogAsync(flatFile.getCommitLogMinOffset(), length).thenApply(MessageBufferUtil::getStoreTimeStamp);
    }

    @Override
    public CompletableFuture<Long> getMessageStoreTimeStampAsync(String topic, int queueId, long queueOffset) {
        CompositeQueueFlatFile flatFile = this.flatFileManager.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            return CompletableFuture.completedFuture(-1L);
        }
        return ((CompletableFuture)((CompletableFuture)flatFile.getConsumeQueueAsync(queueOffset).thenComposeAsync(cqItem -> {
            long commitLogOffset = CQItemBufferUtil.getCommitLogOffset(cqItem);
            int size = CQItemBufferUtil.getSize(cqItem);
            return flatFile.getCommitLogAsync(commitLogOffset, size);
        }, (Executor)TieredStoreExecutor.fetchDataExecutor)).thenApply(MessageBufferUtil::getStoreTimeStamp)).exceptionally(e -> {
            LOGGER.error("TieredMessageFetcher#getMessageStoreTimeStampAsync: get or decode message failed: topic: {}, queue: {}, offset: {}", new Object[]{topic, queueId, queueOffset, e});
            return -1L;
        });
    }

    @Override
    public long getOffsetInQueueByTime(String topic, int queueId, long timestamp, BoundaryType type) {
        CompositeQueueFlatFile flatFile = this.flatFileManager.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            return -1L;
        }
        try {
            return flatFile.getOffsetInConsumeQueueByTime(timestamp, type);
        }
        catch (Exception e) {
            LOGGER.error("TieredMessageFetcher#getOffsetInQueueByTime: get offset in queue by time failed: topic: {}, queue: {}, timestamp: {}, type: {}", new Object[]{topic, queueId, timestamp, type, e});
            return -1L;
        }
    }

    @Override
    public CompletableFuture<QueryMessageResult> queryMessageAsync(String topic, String key, int maxCount, long begin, long end) {
        long topicId;
        IndexService indexStoreService = TieredFlatFileManager.getTieredIndexService(this.storeConfig);
        try {
            TopicMetadata topicMetadata = this.metadataStore.getTopic(topic);
            if (topicMetadata == null) {
                LOGGER.info("MessageFetcher#queryMessageAsync, topic metadata not found, topic: {}", (Object)topic);
                return CompletableFuture.completedFuture(new QueryMessageResult());
            }
            topicId = topicMetadata.getTopicId();
        }
        catch (Exception e) {
            LOGGER.error("MessageFetcher#queryMessageAsync, get topic id failed, topic: {}", (Object)topic, (Object)e);
            return CompletableFuture.completedFuture(new QueryMessageResult());
        }
        CompletableFuture<List<IndexItem>> future = indexStoreService.queryAsync(topic, key, maxCount, begin, end);
        return ((CompletableFuture)future.thenCompose(indexItemList -> {
            QueryMessageResult result = new QueryMessageResult();
            ArrayList<CompletionStage> futureList = new ArrayList<CompletionStage>(maxCount);
            for (IndexItem indexItem : indexItemList) {
                CompositeQueueFlatFile flatFile;
                if (topicId != (long)indexItem.getTopicId() || (flatFile = this.flatFileManager.getFlatFile(new MessageQueue(topic, this.brokerName, indexItem.getQueueId()))) == null) continue;
                CompletionStage getMessageFuture = flatFile.getCommitLogAsync(indexItem.getOffset(), indexItem.getSize()).thenAccept(messageBuffer -> result.addMessage(new SelectMappedBufferResult(indexItem.getOffset(), messageBuffer, indexItem.getSize(), null)));
                futureList.add(getMessageFuture);
                if (futureList.size() < maxCount) continue;
                break;
            }
            return CompletableFuture.allOf(futureList.toArray(new CompletableFuture[0])).thenApply(v -> result);
        })).whenComplete((result, throwable) -> {
            if (result != null) {
                LOGGER.info("MessageFetcher#queryMessageAsync, query result: {}, topic: {}, topicId: {}, key: {}, maxCount: {}, timestamp: {}-{}", new Object[]{result.getMessageBufferList().size(), topic, topicId, key, maxCount, begin, end});
            }
        });
    }
}

