/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.BrokerConfiguration;
import io.moquette.broker.DebugUtils;
import io.moquette.broker.InflightResender;
import io.moquette.broker.LimitedQuota;
import io.moquette.broker.NettyUtils;
import io.moquette.broker.PostOffice;
import io.moquette.broker.Quota;
import io.moquette.broker.Session;
import io.moquette.broker.SessionCorruptedException;
import io.moquette.broker.SessionRegistry;
import io.moquette.broker.TopicAliasMapping;
import io.moquette.broker.Utils;
import io.moquette.broker.security.IAuthenticator;
import io.moquette.broker.security.PemUtils;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.mqtt.MqttConnAckMessage;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttConnectPayload;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageBuilders;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttProperties;
import io.netty.handler.codec.mqtt.MqttPubAckMessage;
import io.netty.handler.codec.mqtt.MqttPubReplyMessageVariableHeader;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttReasonCodeAndPropertiesVariableHeader;
import io.netty.handler.codec.mqtt.MqttReasonCodes;
import io.netty.handler.codec.mqtt.MqttSubAckMessage;
import io.netty.handler.codec.mqtt.MqttSubscribeMessage;
import io.netty.handler.codec.mqtt.MqttUnsubAckMessage;
import io.netty.handler.codec.mqtt.MqttUnsubAckPayload;
import io.netty.handler.codec.mqtt.MqttUnsubscribeMessage;
import io.netty.handler.codec.mqtt.MqttVersion;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.concurrent.GenericFutureListener;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.StandardCharsets;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.net.ssl.SSLPeerUnverifiedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class MQTTConnection {
    private static final Logger LOG = LoggerFactory.getLogger(MQTTConnection.class);
    static final boolean sessionLoopDebug = Boolean.parseBoolean(System.getProperty("moquette.session_loop.debug", "false"));
    private static final int UNDEFINED_VERSION = -1;
    final Channel channel;
    private final BrokerConfiguration brokerConfig;
    private final IAuthenticator authenticator;
    private final SessionRegistry sessionRegistry;
    private final PostOffice postOffice;
    private final int topicAliasMaximum;
    private volatile boolean connected;
    private final AtomicInteger lastPacketId = new AtomicInteger(0);
    private Session bindedSession;
    private int protocolVersion;
    private Quota receivedQuota;
    private Quota sendQuota;
    private TopicAliasMapping aliasMappings;

    MQTTConnection(Channel channel, BrokerConfiguration brokerConfig, IAuthenticator authenticator, SessionRegistry sessionRegistry, PostOffice postOffice) {
        this.channel = channel;
        this.brokerConfig = brokerConfig;
        this.authenticator = authenticator;
        this.sessionRegistry = sessionRegistry;
        this.postOffice = postOffice;
        this.connected = false;
        this.protocolVersion = -1;
        this.topicAliasMaximum = brokerConfig.topicAliasMaximum();
    }

    void handleMessage(MqttMessage msg) {
        MqttMessageType messageType = msg.fixedHeader().messageType();
        LOG.debug("Received MQTT message, type: {}", (Object)messageType);
        switch (messageType) {
            case CONNECT: {
                this.processConnect((MqttConnectMessage)msg);
                break;
            }
            case SUBSCRIBE: {
                this.processSubscribe((MqttSubscribeMessage)msg);
                break;
            }
            case UNSUBSCRIBE: {
                this.processUnsubscribe((MqttUnsubscribeMessage)msg);
                break;
            }
            case PUBLISH: {
                this.processPublish((MqttPublishMessage)msg);
                break;
            }
            case PUBREC: {
                this.processPubRec(msg);
                break;
            }
            case PUBCOMP: {
                this.processPubComp(msg);
                break;
            }
            case PUBREL: {
                this.processPubRel(msg);
                break;
            }
            case DISCONNECT: {
                this.processDisconnect(msg);
                break;
            }
            case PUBACK: {
                this.processPubAck(msg);
                break;
            }
            case PINGREQ: {
                MqttFixedHeader pingHeader = new MqttFixedHeader(MqttMessageType.PINGRESP, false, MqttQoS.AT_MOST_ONCE, false, 0);
                MqttMessage pingResp = new MqttMessage(pingHeader);
                this.channel.writeAndFlush((Object)pingResp).addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
                break;
            }
            default: {
                LOG.error("Unknown MessageType: {}", (Object)messageType);
            }
        }
    }

    private void processPubComp(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        String clientID = this.bindedSession.getClientID();
        this.postOffice.routeCommand(clientID, "PUBCOMP", () -> {
            this.checkMatchSessionLoop(clientID);
            this.bindedSession.processPubComp(messageID);
            return null;
        });
    }

    private void processPubRec(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        String clientID = this.bindedSession.getClientID();
        this.postOffice.routeCommand(clientID, "PUBREC", () -> {
            this.checkMatchSessionLoop(clientID);
            this.bindedSession.processPubRec(messageID);
            return null;
        });
    }

    static MqttMessage pubrel(int messageID) {
        MqttFixedHeader pubRelHeader = new MqttFixedHeader(MqttMessageType.PUBREL, false, MqttQoS.AT_LEAST_ONCE, false, 0);
        return new MqttMessage(pubRelHeader, (Object)MqttMessageIdVariableHeader.from((int)messageID));
    }

    private void processPubAck(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        String clientId = this.getClientId();
        this.postOffice.routeCommand(clientId, "PUB ACK", () -> {
            this.checkMatchSessionLoop(clientId);
            this.bindedSession.pubAckReceived(messageID);
            return null;
        });
    }

    public static Quota createQuota(int receiveMaximum) {
        return new LimitedQuota(receiveMaximum);
    }

    Quota sendQuota() {
        return this.sendQuota;
    }

    PostOffice.RouteResult processConnect(MqttConnectMessage msg) {
        boolean serverGeneratedClientId;
        MqttConnectPayload payload = msg.payload();
        String clientId = payload.clientIdentifier();
        String username = payload.userName();
        LOG.trace("Processing CONNECT message. CId: {} username: {}", (Object)clientId, (Object)username);
        if (MQTTConnection.isNotProtocolVersion(msg, MqttVersion.MQTT_3_1) && MQTTConnection.isNotProtocolVersion(msg, MqttVersion.MQTT_3_1_1) && MQTTConnection.isNotProtocolVersion(msg, MqttVersion.MQTT_5)) {
            LOG.warn("MQTT protocol version is not valid. CId: {}", (Object)clientId);
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_UNACCEPTABLE_PROTOCOL_VERSION);
            return PostOffice.RouteResult.failed(clientId);
        }
        boolean cleanSession = msg.variableHeader().isCleanSession();
        if (clientId == null || clientId.isEmpty()) {
            if (MQTTConnection.isNotProtocolVersion(msg, MqttVersion.MQTT_5)) {
                if (!this.brokerConfig.isAllowZeroByteClientId()) {
                    LOG.info("Broker doesn't permit MQTT empty client ID. Username: {}", (Object)username);
                    this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED);
                    return PostOffice.RouteResult.failed(clientId);
                }
                if (!cleanSession) {
                    LOG.info("MQTT client ID cannot be empty for persistent session. Username: {}", (Object)username);
                    this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED);
                    return PostOffice.RouteResult.failed(clientId);
                }
            }
            clientId = UUID.randomUUID().toString().replace("-", "");
            serverGeneratedClientId = true;
            LOG.debug("Client has connected with integration generated id: {}, username: {}", (Object)clientId, (Object)username);
        } else {
            serverGeneratedClientId = false;
        }
        if (!this.login(msg, clientId)) {
            if (MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_5)) {
                MqttMessageBuilders.ConnAckPropertiesBuilder builder = this.prepareConnAckPropertiesBuilder(false, clientId);
                builder.reasonString("User credentials provided are not recognized as valid");
                this.abortConnectionV5(MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USERNAME_OR_PASSWORD, builder);
            } else {
                this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USER_NAME_OR_PASSWORD);
            }
            this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
            return PostOffice.RouteResult.failed(clientId);
        }
        if (MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_5)) {
            this.aliasMappings = new TopicAliasMapping();
        }
        this.receivedQuota = MQTTConnection.createQuota(this.brokerConfig.receiveMaximum());
        this.sendQuota = this.retrieveSendQuota(msg);
        String sessionId = clientId;
        this.protocolVersion = msg.variableHeader().version();
        return this.postOffice.routeCommand(clientId, "CONN", () -> {
            this.checkMatchSessionLoop(sessionId);
            this.executeConnect(msg, sessionId, serverGeneratedClientId);
            return null;
        });
    }

    private Quota retrieveSendQuota(MqttConnectMessage msg) {
        if (MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_3_1) || MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_3_1_1)) {
            return MQTTConnection.createQuota(10);
        }
        MqttProperties.IntegerProperty receiveMaximumProperty = (MqttProperties.IntegerProperty)msg.variableHeader().properties().getProperty(MqttProperties.MqttPropertyType.RECEIVE_MAXIMUM.value());
        if (receiveMaximumProperty == null) {
            return MQTTConnection.createQuota(66560);
        }
        return MQTTConnection.createQuota((Integer)receiveMaximumProperty.value());
    }

    protected void assignSendQuota(Quota quota) {
        this.sendQuota = quota;
    }

    private void checkMatchSessionLoop(String clientId) {
        if (!sessionLoopDebug) {
            return;
        }
        String currentThreadName = Thread.currentThread().getName();
        String expectedThreadName = this.postOffice.sessionLoopThreadName(clientId);
        if (!expectedThreadName.equals(currentThreadName)) {
            throw new IllegalStateException("Expected to be executed on thread " + expectedThreadName + " but running on " + currentThreadName + ". This means a programming error");
        }
    }

    private void executeConnect(final MqttConnectMessage msg, String clientId, boolean serverGeneratedClientId) {
        SessionRegistry.SessionCreationResult result;
        byte[] willPayload;
        boolean validWillPayload;
        boolean hasPayloadFormatIndicator;
        if (MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_5) && (hasPayloadFormatIndicator = MQTTConnection.extractWillPayloadFormatIndicator(msg.payload().willProperties())) && !(validWillPayload = MQTTConnection.checkUTF8Validity(willPayload = msg.payload().willMessageInBytes()))) {
            MqttMessageBuilders.ConnAckPropertiesBuilder builder = this.prepareConnAckPropertiesBuilder(false, clientId);
            builder.reasonString("Not UTF8 payload in Will");
            this.abortConnectionV5(MqttConnectReturnCode.CONNECTION_REFUSED_PAYLOAD_FORMAT_INVALID, builder);
            return;
        }
        try {
            LOG.trace("Binding MQTTConnection to session");
            result = this.sessionRegistry.createOrReopenSession(msg, clientId, this.getUsername());
            result.session.bind(this);
            this.bindedSession = result.session;
        }
        catch (SessionCorruptedException scex) {
            LOG.warn("MQTT session for client ID {} cannot be created", (Object)clientId);
            if (MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_5)) {
                MqttMessageBuilders.ConnAckPropertiesBuilder builder = this.prepareConnAckPropertiesBuilder(false, clientId);
                builder.reasonString("Error creating the session, retry later");
                this.abortConnectionV5(MqttConnectReturnCode.CONNECTION_REFUSED_CLIENT_IDENTIFIER_NOT_VALID, builder);
            } else {
                this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_SERVER_UNAVAILABLE);
            }
            return;
        }
        NettyUtils.clientID(this.channel, clientId);
        boolean msgCleanSessionFlag = msg.variableHeader().isCleanSession();
        boolean isSessionAlreadyPresent = !msgCleanSessionFlag && result.alreadyStored;
        final String clientIdUsed = clientId;
        MqttMessageBuilders.ConnAckBuilder connAckBuilder = MqttMessageBuilders.connAck().returnCode(MqttConnectReturnCode.CONNECTION_ACCEPTED).sessionPresent(isSessionAlreadyPresent);
        if (MQTTConnection.isProtocolVersion(msg, MqttVersion.MQTT_5)) {
            MqttMessageBuilders.ConnAckPropertiesBuilder connAckPropertiesBuilder = this.prepareConnAckPropertiesBuilder(serverGeneratedClientId, clientId);
            if (MQTTConnection.isNeedResponseInformation(msg)) {
                connAckPropertiesBuilder.responseInformation("/reqresp/response/" + clientId);
            }
            if (this.receivedQuota.hasLimit()) {
                connAckPropertiesBuilder.receiveMaximum(this.receivedQuota.getMaximum());
            }
            if (this.topicAliasMaximum != 0) {
                connAckPropertiesBuilder.topicAliasMaximum(this.topicAliasMaximum);
            }
            if (this.brokerConfig.getServerKeepAlive().isPresent()) {
                connAckPropertiesBuilder.serverKeepAlive(this.brokerConfig.getServerKeepAlive().get().intValue());
            }
            MqttProperties ackProperties = connAckPropertiesBuilder.build();
            connAckBuilder.properties(ackProperties);
        }
        MqttConnAckMessage ackMessage = connAckBuilder.build();
        this.channel.writeAndFlush((Object)ackMessage).addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    LOG.trace("CONNACK sent, channel: {}", (Object)MQTTConnection.this.channel);
                    if (!result.session.completeConnection()) {
                        MqttMessage disconnectMsg = MqttMessageBuilders.disconnect().build();
                        MQTTConnection.this.channel.writeAndFlush((Object)disconnectMsg).addListener((GenericFutureListener)CLOSE);
                        LOG.warn("CONNACK is sent but the session created can't transition in CONNECTED state");
                    } else {
                        MQTTConnection.this.connected = true;
                        MQTTConnection.this.postOffice.wipeExistingScheduledWill(clientIdUsed);
                        if (result.mode == SessionRegistry.CreationModeEnum.REOPEN_EXISTING) {
                            Session session = result.session;
                            MQTTConnection.this.postOffice.routeCommand(session.getClientID(), "sendOfflineMessages", () -> {
                                session.reconnectSession();
                                return null;
                            });
                        }
                        MQTTConnection.this.initializeKeepAliveTimeout(MQTTConnection.this.channel, msg, clientIdUsed);
                        if (MQTTConnection.isNotProtocolVersion(msg, MqttVersion.MQTT_5)) {
                            MQTTConnection.this.setupInflightResender(MQTTConnection.this.channel);
                        }
                        MQTTConnection.this.postOffice.dispatchConnection(msg);
                        LOG.trace("dispatch connection: {}", (Object)msg);
                    }
                } else {
                    MQTTConnection.this.sessionRegistry.connectionClosed(MQTTConnection.this.bindedSession);
                    LOG.error("CONNACK send failed, cleanup session and close the connection", future.cause());
                    MQTTConnection.this.channel.close();
                }
            }
        });
    }

    static boolean isNeedResponseInformation(MqttConnectMessage msg) {
        MqttProperties.IntegerProperty requestRespInfo = (MqttProperties.IntegerProperty)msg.variableHeader().properties().getProperty(MqttProperties.MqttPropertyType.REQUEST_RESPONSE_INFORMATION.value());
        return requestRespInfo != null && (Integer)requestRespInfo.value() >= 1;
    }

    private static boolean extractWillPayloadFormatIndicator(MqttProperties mqttProperties) {
        MqttProperties.MqttProperty payloadFormatIndicatorProperty = mqttProperties.getProperty(MqttProperties.MqttPropertyType.PAYLOAD_FORMAT_INDICATOR.value());
        boolean hasPayloadFormatIndicator = false;
        if (payloadFormatIndicatorProperty != null) {
            int payloadFormatIndicator = (Integer)payloadFormatIndicatorProperty.value();
            hasPayloadFormatIndicator = payloadFormatIndicator == 1;
        }
        return hasPayloadFormatIndicator;
    }

    private static boolean checkUTF8Validity(byte[] rawBytes) {
        CharsetDecoder decoder = StandardCharsets.UTF_8.newDecoder();
        try {
            decoder.decode(ByteBuffer.wrap(rawBytes));
        }
        catch (CharacterCodingException ex) {
            return false;
        }
        return true;
    }

    private MqttMessageBuilders.ConnAckPropertiesBuilder prepareConnAckPropertiesBuilder(boolean serverGeneratedClientId, String clientId) {
        MqttMessageBuilders.ConnAckPropertiesBuilder builder = new MqttMessageBuilders.ConnAckPropertiesBuilder();
        if (serverGeneratedClientId) {
            builder.assignedClientId(clientId);
        }
        builder.sessionExpiryInterval(-1L).retainAvailable(true).wildcardSubscriptionAvailable(true).subscriptionIdentifiersAvailable(true).sharedSubscriptionAvailable(true);
        return builder;
    }

    private void setupInflightResender(Channel channel) {
        channel.pipeline().addFirst("inflightResender", (ChannelHandler)new InflightResender(5000L, TimeUnit.MILLISECONDS));
    }

    private void initializeKeepAliveTimeout(Channel channel, MqttConnectMessage msg, String clientId) {
        int keepAlive = msg.variableHeader().keepAliveTimeSeconds();
        if (this.brokerConfig.getServerKeepAlive().isPresent()) {
            int serverKeepAlive = this.brokerConfig.getServerKeepAlive().get();
            LOG.info("Forcing server keep alive ({}) over client selection ({})", (Object)serverKeepAlive, (Object)keepAlive);
            keepAlive = serverKeepAlive;
        }
        NettyUtils.keepAlive(channel, keepAlive);
        NettyUtils.cleanSession(channel, msg.variableHeader().isCleanSession());
        NettyUtils.clientID(channel, clientId);
        int idleTime = Math.round((float)keepAlive * 1.5f);
        this.setIdleTime(channel.pipeline(), idleTime);
        LOG.debug("Connection has been configured CId={}, keepAlive={}, removeTemporaryQoS2={}, idleTime={}", new Object[]{clientId, keepAlive, msg.variableHeader().isCleanSession(), idleTime});
    }

    private void setIdleTime(ChannelPipeline pipeline, int idleTime) {
        if (pipeline.names().contains("idleStateHandler")) {
            pipeline.remove("idleStateHandler");
        }
        pipeline.addFirst("idleStateHandler", (ChannelHandler)new IdleStateHandler(idleTime, 0, 0));
    }

    private static boolean isNotProtocolVersion(MqttConnectMessage msg, MqttVersion version) {
        return !MQTTConnection.isProtocolVersion(msg, version);
    }

    private static boolean isProtocolVersion(MqttConnectMessage msg, MqttVersion version) {
        return msg.variableHeader().version() == version.protocolLevel();
    }

    private void abortConnection(MqttConnectReturnCode returnCode) {
        MqttConnAckMessage badProto = MqttMessageBuilders.connAck().returnCode(returnCode).sessionPresent(false).build();
        this.channel.writeAndFlush((Object)badProto).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
    }

    private void abortConnectionV5(MqttConnectReturnCode returnCode, MqttMessageBuilders.ConnAckPropertiesBuilder propertiesBuilder) {
        MqttConnAckMessage badProto = MqttMessageBuilders.connAck().returnCode(returnCode).properties(propertiesBuilder.build()).sessionPresent(false).build();
        this.channel.writeAndFlush((Object)badProto).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
    }

    private boolean login(MqttConnectMessage msg, String clientId) {
        String userName = null;
        byte[] pwd = null;
        if (msg.variableHeader().hasUserName()) {
            userName = msg.payload().userName();
            if (msg.variableHeader().hasPassword()) {
                pwd = msg.payload().passwordInBytes();
            }
        }
        if (this.brokerConfig.isPeerCertificateAsUsername()) {
            userName = this.readClientProvidedCertificates(clientId);
        }
        if (userName == null || userName.isEmpty()) {
            if (this.brokerConfig.isAllowAnonymous()) {
                return true;
            }
            LOG.info("Client didn't supply any credentials and MQTT anonymous mode is disabled. CId={}", (Object)clientId);
            return false;
        }
        if (!this.authenticator.checkValid(clientId, userName, pwd)) {
            LOG.info("Authenticator has rejected the MQTT credentials CId={}, username={}", (Object)clientId, (Object)userName);
            return false;
        }
        NettyUtils.userName(this.channel, userName);
        return true;
    }

    private String readClientProvidedCertificates(String clientId) {
        try {
            SslHandler sslhandler = (SslHandler)this.channel.pipeline().get("ssl");
            if (sslhandler != null) {
                Certificate[] certificateChain = sslhandler.engine().getSession().getPeerCertificates();
                return PemUtils.certificatesToPem(certificateChain);
            }
        }
        catch (SSLPeerUnverifiedException e) {
            LOG.debug("No peer cert provided. CId={}", (Object)clientId);
        }
        catch (IOException | CertificateEncodingException e) {
            LOG.warn("Unable to decode client certificate. CId={}", (Object)clientId);
        }
        return null;
    }

    void handleConnectionLost() {
        String clientID = NettyUtils.clientID(this.channel);
        if (clientID == null || clientID.isEmpty()) {
            return;
        }
        LOG.debug("Notifying connection lost event");
        this.postOffice.routeCommand(clientID, "CONN LOST", () -> {
            this.checkMatchSessionLoop(clientID);
            if (this.isBoundToSession() || this.isSessionUnbound()) {
                LOG.debug("Cleaning {}", (Object)clientID);
                this.processConnectionLost(clientID);
            } else {
                LOG.debug("NOT Cleaning {}, bound to other connection.", (Object)clientID);
            }
            return null;
        });
    }

    private void processConnectionLost(String clientID) {
        if (this.bindedSession.hasWill()) {
            this.postOffice.fireWill(this.bindedSession);
        }
        if (this.bindedSession.connected()) {
            LOG.debug("Closing session on connectionLost {}", (Object)clientID);
            this.sessionRegistry.connectionClosed(this.bindedSession);
            this.connected = false;
        }
        String userName = NettyUtils.userName(this.channel);
        this.postOffice.dispatchConnectionLost(clientID, userName);
        LOG.trace("dispatch disconnection: userName={}", (Object)userName);
    }

    boolean isConnected() {
        return this.connected;
    }

    void dropConnection() {
        this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    PostOffice.RouteResult processDisconnect(MqttMessage msg) {
        String clientID = NettyUtils.clientID(this.channel);
        LOG.trace("Start DISCONNECT");
        if (!this.connected) {
            LOG.info("DISCONNECT received on already closed connection");
            return PostOffice.RouteResult.success(clientID, CompletableFuture.completedFuture(null));
        }
        return this.postOffice.routeCommand(clientID, "DISCONN", () -> {
            MqttReasonCodeAndPropertiesVariableHeader disconnectHeader;
            this.checkMatchSessionLoop(clientID);
            if (!this.isBoundToSession()) {
                LOG.debug("NOT processing disconnect {}, not bound.", (Object)clientID);
                return null;
            }
            if (this.isProtocolVersion5() && (disconnectHeader = (MqttReasonCodeAndPropertiesVariableHeader)msg.variableHeader()).reasonCode() != MqttReasonCodes.Disconnect.NORMAL_DISCONNECT.byteValue() && this.bindedSession.hasWill()) {
                this.postOffice.fireWill(this.bindedSession);
            }
            LOG.debug("Closing session on disconnect {}", (Object)clientID);
            this.sessionRegistry.connectionClosed(this.bindedSession);
            this.connected = false;
            this.protocolVersion = -1;
            this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
            String userName = NettyUtils.userName(this.channel);
            this.postOffice.clientDisconnected(clientID, userName);
            LOG.trace("dispatch disconnection userName={}", (Object)userName);
            return null;
        });
    }

    boolean isProtocolVersion5() {
        return this.protocolVersion == MqttVersion.MQTT_5.protocolLevel();
    }

    PostOffice.RouteResult processSubscribe(MqttSubscribeMessage msg) {
        String clientID = NettyUtils.clientID(this.channel);
        if (!this.connected) {
            LOG.warn("SUBSCRIBE received on already closed connection");
            this.dropConnection();
            return PostOffice.RouteResult.success(clientID, CompletableFuture.completedFuture(null));
        }
        String username = NettyUtils.userName(this.channel);
        return this.postOffice.routeCommand(clientID, "SUB", () -> {
            this.checkMatchSessionLoop(clientID);
            if (this.isBoundToSession()) {
                this.postOffice.subscribeClientToTopics(msg, clientID, username, this);
            }
            return null;
        });
    }

    void sendSubAckMessage(int messageID, MqttSubAckMessage ackMessage) {
        LOG.trace("Sending SUBACK response messageId: {}", (Object)messageID);
        this.channel.writeAndFlush((Object)ackMessage).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    private void processUnsubscribe(MqttUnsubscribeMessage msg) {
        List topics = msg.payload().topics();
        String clientID = NettyUtils.clientID(this.channel);
        int messageId = msg.variableHeader().messageId();
        this.postOffice.routeCommand(clientID, "UNSUB", () -> {
            this.checkMatchSessionLoop(clientID);
            if (!this.isBoundToSession()) {
                return null;
            }
            LOG.trace("Processing UNSUBSCRIBE message. topics: {}", (Object)topics);
            this.postOffice.unsubscribe(topics, this, messageId);
            return null;
        });
    }

    void sendUnsubAckMessage(List<String> topics, String clientID, int messageID) {
        MqttUnsubAckMessage ackMessage;
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.UNSUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        if (this.isProtocolVersion5()) {
            MqttUnsubAckPayload payload = new MqttUnsubAckPayload(new short[]{MqttReasonCodes.UnsubAck.SUCCESS.byteValue()});
            ackMessage = new MqttUnsubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID), payload);
        } else {
            ackMessage = new MqttUnsubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID));
        }
        LOG.trace("Sending UNSUBACK message. messageId: {}, topics: {}", (Object)messageID, topics);
        this.channel.writeAndFlush((Object)ackMessage).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        LOG.trace("Client unsubscribed from topics <{}>", topics);
    }

    PostOffice.RouteResult processPublish(MqttPublishMessage msg) {
        MqttProperties.MqttProperty topicAlias;
        MqttQoS qos = msg.fixedHeader().qosLevel();
        String username = NettyUtils.userName(this.channel);
        String topicName = msg.variableHeader().topicName();
        String clientId = this.getClientId();
        int messageID = msg.variableHeader().packetId();
        LOG.trace("Processing PUBLISH message, topic: {}, messageId: {}, qos: {}", new Object[]{topicName, messageID, qos});
        Topic topic = new Topic(topicName);
        if (this.isProtocolVersion5() && (topicAlias = msg.variableHeader().properties().getProperty(MqttProperties.MqttPropertyType.TOPIC_ALIAS.value())) != null) {
            try {
                Optional<String> mappedTopicName = this.updateAndMapTopicAlias((MqttProperties.IntegerProperty)topicAlias, topicName);
                String translatedTopicName = mappedTopicName.orElse(topicName);
                msg = MQTTConnection.copyPublishMessageExceptTopicAlias(msg, translatedTopicName, messageID);
                topic = new Topic(translatedTopicName);
            }
            catch (ErrorCodeException e) {
                this.brokerDisconnect(e.getErrorCode());
                this.disconnectSession();
                this.dropConnection();
                return PostOffice.RouteResult.failed(clientId);
            }
        }
        if (!topic.isValid()) {
            LOG.debug("Drop connection because of invalid topic format");
            this.dropConnection();
        }
        if (!topic.isEmpty() && topic.headToken().isReserved()) {
            LOG.warn("Avoid to publish on topic which contains reserved topic (starts with $)");
            return PostOffice.RouteResult.failed(clientId);
        }
        Instant expiry = this.extractExpiryFromProperty(msg);
        Utils.retain(msg, "PUB in");
        MqttPublishMessage finalMsg = msg;
        switch (qos) {
            case AT_MOST_ONCE: {
                return this.postOffice.routeCommand(clientId, "PUB QoS0", () -> {
                    this.checkMatchSessionLoop(clientId);
                    if (!this.isBoundToSession()) {
                        return null;
                    }
                    this.postOffice.receivedPublishQos0(this, username, clientId, finalMsg, expiry);
                    return null;
                }).ifFailed(() -> Utils.release(finalMsg, "PUB in - failed"));
            }
            case AT_LEAST_ONCE: {
                if (!this.receivedQuota.hasFreeSlots()) {
                    LOG.warn("Client {} exceeded the quota {} processing QoS1, disconnecting it", (Object)clientId, (Object)this.receivedQuota);
                    Utils.release(msg, "PUB in - QoS1 exceeded quota");
                    this.brokerDisconnect(MqttReasonCodes.Disconnect.RECEIVE_MAXIMUM_EXCEEDED);
                    this.disconnectSession();
                    this.dropConnection();
                    return null;
                }
                return this.postOffice.routeCommand(clientId, "PUB QoS1", () -> {
                    this.checkMatchSessionLoop(clientId);
                    if (!this.isBoundToSession()) {
                        return null;
                    }
                    this.receivedQuota.consumeSlot();
                    this.postOffice.receivedPublishQos1(this, username, messageID, finalMsg, expiry).completableFuture().thenRun(() -> this.receivedQuota.releaseSlot());
                    return null;
                }).ifFailed(() -> Utils.release(finalMsg, "PUB in - failed"));
            }
            case EXACTLY_ONCE: {
                if (!this.receivedQuota.hasFreeSlots()) {
                    LOG.warn("Client {} exceeded the quota {} processing QoS2, disconnecting it", (Object)clientId, (Object)this.receivedQuota);
                    Utils.release(finalMsg, "PUB in - phase 1 QoS2 exceeded quota");
                    this.brokerDisconnect(MqttReasonCodes.Disconnect.RECEIVE_MAXIMUM_EXCEEDED);
                    this.disconnectSession();
                    this.dropConnection();
                    return null;
                }
                PostOffice.RouteResult firstStepResult = this.postOffice.routeCommand(clientId, "PUB QoS2", () -> {
                    this.checkMatchSessionLoop(clientId);
                    if (!this.isBoundToSession()) {
                        return null;
                    }
                    this.bindedSession.receivedPublishQos2(messageID, finalMsg);
                    this.receivedQuota.consumeSlot();
                    return null;
                });
                if (!firstStepResult.isSuccess()) {
                    Utils.release(msg, "PUB in - failed");
                    LOG.trace("Failed to enqueue PUB QoS2 to session loop for {}", (Object)clientId);
                    return firstStepResult;
                }
                firstStepResult.completableFuture().thenRun(() -> this.postOffice.receivedPublishQos2(this, finalMsg, username, expiry).completableFuture());
                return firstStepResult;
            }
        }
        LOG.error("Unknown QoS-Type:{}", (Object)qos);
        return PostOffice.RouteResult.failed(clientId, "Unknown QoS-");
    }

    private Optional<String> updateAndMapTopicAlias(MqttProperties.IntegerProperty topicAlias, String topicName) throws ErrorCodeException {
        if (this.topicAliasMaximum == 0) {
            LOG.info("Dropping connection {}, received a PUBLISH with topic alias while the feature is not enaled on the broker", (Object)this.channel);
            throw new ErrorCodeException(MqttReasonCodes.Disconnect.PROTOCOL_ERROR);
        }
        MqttProperties.IntegerProperty topicAliasTyped = topicAlias;
        if ((Integer)topicAliasTyped.value() == 0 || (Integer)topicAliasTyped.value() > this.topicAliasMaximum) {
            LOG.info("Dropping connection {}, received a topic alias ({}) outside of range (0..{}]", new Object[]{this.channel, topicAliasTyped.value(), this.topicAliasMaximum});
            throw new ErrorCodeException(MqttReasonCodes.Disconnect.TOPIC_ALIAS_INVALID);
        }
        Optional<String> mappedTopicName = this.aliasMappings.topicFromAlias((Integer)topicAliasTyped.value());
        if (mappedTopicName.isPresent()) {
            if (topicName != null) {
                this.aliasMappings.update(topicName, (Integer)topicAliasTyped.value());
            }
        } else {
            if (topicName.isEmpty()) {
                throw new ErrorCodeException(MqttReasonCodes.Disconnect.PROTOCOL_ERROR);
            }
            this.aliasMappings.update(topicName, (Integer)topicAliasTyped.value());
        }
        return mappedTopicName;
    }

    private static MqttPublishMessage copyPublishMessageExceptTopicAlias(MqttPublishMessage msg, String translatedTopicName, int messageID) {
        MqttProperties rewrittenProps = new MqttProperties();
        for (MqttProperties.MqttProperty property : msg.variableHeader().properties().listAll()) {
            if (property.propertyId() == MqttProperties.MqttPropertyType.TOPIC_ALIAS.value()) continue;
            rewrittenProps.add(property);
        }
        MqttPublishVariableHeader rewrittenVariableHeader = new MqttPublishVariableHeader(translatedTopicName, messageID, rewrittenProps);
        return new MqttPublishMessage(msg.fixedHeader(), rewrittenVariableHeader, msg.payload());
    }

    private Instant extractExpiryFromProperty(MqttPublishMessage msg) {
        MqttProperties.MqttProperty expiryProp = msg.variableHeader().properties().getProperty(MqttProperties.MqttPropertyType.PUBLICATION_EXPIRY_INTERVAL.value());
        if (expiryProp == null) {
            return Instant.MAX;
        }
        Integer expirySeconds = (Integer)((MqttProperties.IntegerProperty)expiryProp).value();
        return Instant.now().plusSeconds(expirySeconds.intValue());
    }

    void sendPubRec(int messageID) {
        LOG.trace("sendPubRec invoked, messageID: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBREC, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubAckMessage pubRecMessage = new MqttPubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID));
        this.sendIfWritableElseDrop((MqttMessage)pubRecMessage);
    }

    void sendPubRec(int messageID, MqttReasonCodes.PubRec reasonCode) {
        LOG.trace("sendPubRec for messageID: {}, reason code: {}", (Object)messageID, (Object)reasonCode);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBREC, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubReplyMessageVariableHeader variableHeader = new MqttPubReplyMessageVariableHeader(messageID, reasonCode.byteValue(), MqttProperties.NO_PROPERTIES);
        MqttPubAckMessage pubRecMessage = new MqttPubAckMessage(fixedHeader, (MqttMessageIdVariableHeader)variableHeader);
        this.sendIfWritableElseDrop((MqttMessage)pubRecMessage);
    }

    private void processPubRel(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        String clientID = this.bindedSession.getClientID();
        this.postOffice.routeCommand(clientID, "PUBREL", () -> {
            this.checkMatchSessionLoop(clientID);
            this.executePubRel(messageID);
            this.receivedQuota.releaseSlot();
            return null;
        });
    }

    private void executePubRel(int messageID) {
        this.bindedSession.receivedPubRelQos2(messageID);
        this.sendPubCompMessage(messageID);
    }

    void sendPublish(MqttPublishMessage publishMsg) {
        int packetId = publishMsg.variableHeader().packetId();
        String topicName = publishMsg.variableHeader().topicName();
        MqttQoS qos = publishMsg.fixedHeader().qosLevel();
        if (LOG.isTraceEnabled()) {
            LOG.trace("Sending PUBLISH({}) message. MessageId={}, topic={}, payload={} to {}", new Object[]{qos, packetId, topicName, DebugUtils.payload2Str(publishMsg.payload()), this.getClientId()});
        } else {
            LOG.debug("Sending PUBLISH({}) message. MessageId={}, topic={} to {}", new Object[]{qos, packetId, topicName, this.getClientId()});
        }
        this.sendIfWritableElseDrop((MqttMessage)publishMsg);
    }

    void sendIfWritableElseDrop(MqttMessage msg) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("OUT {}", (Object)msg.fixedHeader().messageType());
        }
        if (this.channel.isWritable()) {
            LOG.debug("Sending message {} on the wire to {}", (Object)msg.fixedHeader().messageType(), (Object)this.getClientId());
            MqttMessage retainedDup = msg;
            if (msg instanceof ByteBufHolder) {
                retainedDup = Utils.retainDuplicate((ByteBufHolder)msg, "mqtt connection send PUB");
            }
            ChannelFuture channelFuture = this.brokerConfig.getBufferFlushMillis() == 0 ? this.channel.writeAndFlush((Object)retainedDup) : this.channel.write((Object)retainedDup);
            channelFuture.addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        } else {
            LOG.debug("Dropping message {} from the wire, msg: {}", (Object)msg.fixedHeader().messageType(), (Object)msg);
        }
    }

    public void writabilityChanged() {
        if (this.channel.isWritable()) {
            LOG.debug("Channel is again writable");
            this.postOffice.routeCommand(this.getClientId(), "writabilityChanged", () -> {
                this.bindedSession.writabilityChanged();
                return null;
            });
        }
    }

    void sendPubAck(int messageID) {
        LOG.trace("sendPubAck for messageID: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubAckMessage pubAckMessage = new MqttPubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID));
        this.sendIfWritableElseDrop((MqttMessage)pubAckMessage);
    }

    void sendPubAck(int messageID, MqttReasonCodes.PubAck reasonCode) {
        LOG.trace("sendPubAck for messageID: {}, reason code: {}", (Object)messageID, (Object)reasonCode);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubReplyMessageVariableHeader variableHeader = new MqttPubReplyMessageVariableHeader(messageID, reasonCode.byteValue(), MqttProperties.NO_PROPERTIES);
        MqttPubAckMessage pubAckMessage = new MqttPubAckMessage(fixedHeader, (MqttMessageIdVariableHeader)variableHeader);
        this.sendIfWritableElseDrop((MqttMessage)pubAckMessage);
    }

    private void sendPubCompMessage(int messageID) {
        LOG.trace("Sending PUBCOMP message messageId: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBCOMP, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttMessage pubCompMessage = new MqttMessage(fixedHeader, (Object)MqttMessageIdVariableHeader.from((int)messageID));
        this.sendIfWritableElseDrop(pubCompMessage);
    }

    String getClientId() {
        return NettyUtils.clientID(this.channel);
    }

    String getUsername() {
        return NettyUtils.userName(this.channel);
    }

    static MqttPublishMessage createNotRetainedPublishMessage(String topic, MqttQoS qos, ByteBuf message, int messageId, MqttProperties.MqttProperty ... mqttProperties) {
        return MQTTConnection.createPublishMessage(topic, qos, message, messageId, false, false, mqttProperties);
    }

    static MqttPublishMessage createPublishMessage(String topic, MqttQoS qos, ByteBuf message, int messageId, boolean retained, boolean isDup, MqttProperties.MqttProperty ... mqttProperties) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, isDup, qos, retained, 0);
        MqttProperties props = new MqttProperties();
        for (MqttProperties.MqttProperty mqttProperty : mqttProperties) {
            props.add(mqttProperty);
        }
        MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(topic, messageId, props);
        return new MqttPublishMessage(fixedHeader, varHeader, message);
    }

    static MqttPublishMessage createNotRetainedDuplicatedPublishMessage(int packetId, Topic topic, MqttQoS qos, ByteBuf payload, MqttProperties.MqttProperty ... mqttProperties) {
        return MQTTConnection.createPublishMessage(topic.toString(), qos, payload, packetId, false, true, mqttProperties);
    }

    public void resendNotAckedPublishes() {
        this.bindedSession.resendInflightNotAcked();
    }

    int nextPacketId() {
        return this.lastPacketId.updateAndGet(v -> v == 65535 ? 1 : v + 1);
    }

    public String toString() {
        return "MQTTConnection{channel=" + this.channel + ", connected=" + this.connected + '}';
    }

    InetSocketAddress remoteAddress() {
        return (InetSocketAddress)this.channel.remoteAddress();
    }

    public void readCompleted() {
        LOG.debug("readCompleted client CId: {}", (Object)this.getClientId());
        if (this.getClientId() != null) {
            this.queueDrainQueueCommand();
        }
    }

    private void queueDrainQueueCommand() {
        this.postOffice.routeCommand(this.getClientId(), "flushQueues", () -> {
            this.bindedSession.flushAllQueuedMessages();
            return null;
        });
    }

    public void flush() {
        this.channel.flush();
    }

    private boolean isBoundToSession() {
        return this.bindedSession != null && this.bindedSession.isBoundTo(this);
    }

    private boolean isSessionUnbound() {
        return this.bindedSession != null && this.bindedSession.isBoundTo(null);
    }

    public void bindSession(Session session) {
        this.bindedSession = session;
    }

    void brokerDisconnect() {
        MqttMessage disconnectMsg = MqttMessageBuilders.disconnect().build();
        this.channel.writeAndFlush((Object)disconnectMsg).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
    }

    void brokerDisconnect(MqttReasonCodes.Disconnect reasonCode) {
        this.connected = false;
        MqttMessage disconnectMsg = MqttMessageBuilders.disconnect().reasonCode(reasonCode.byteValue()).build();
        this.channel.writeAndFlush((Object)disconnectMsg).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
    }

    void disconnectSession() {
        this.bindedSession.disconnect();
    }

    static final class ErrorCodeException
    extends Exception {
        private final String hexErrorCode;
        private final MqttReasonCodes.Disconnect errorCode;

        public ErrorCodeException(MqttReasonCodes.Disconnect disconnectError) {
            this.errorCode = disconnectError;
            this.hexErrorCode = Integer.toHexString(disconnectError.byteValue());
        }

        public MqttReasonCodes.Disconnect getErrorCode() {
            return this.errorCode;
        }
    }
}

