/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.utils.sort;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import org.apache.iotdb.commons.exception.IoTDBException;
import org.apache.iotdb.db.utils.datastructure.SortKey;
import org.apache.iotdb.db.utils.sort.FileSpillerReader;
import org.apache.iotdb.db.utils.sort.SortBufferManager;
import org.apache.iotdb.db.utils.sort.SortReader;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.tsfile.block.column.ColumnBuilder;
import org.apache.tsfile.enums.TSDataType;
import org.apache.tsfile.read.common.block.TsBlock;
import org.apache.tsfile.read.common.block.TsBlockBuilder;
import org.apache.tsfile.read.common.block.column.TimeColumnBuilder;
import org.apache.tsfile.read.common.block.column.TsBlockSerde;

public class DiskSpiller {
    private static final String FILE_SUFFIX = ".sortTemp";
    private final List<TSDataType> dataTypeList;
    private final String folderPath;
    private final String filePrefix;
    private int fileIndex;
    private boolean folderCreated = false;
    private final TsBlockSerde serde = new TsBlockSerde();

    public DiskSpiller(String folderPath, String filePrefix, List<TSDataType> dataTypeList) {
        this.folderPath = folderPath;
        this.filePrefix = filePrefix + "-";
        this.fileIndex = 0;
        this.dataTypeList = dataTypeList;
    }

    private void createFolder(String folderPath) throws IOException {
        Path path = Paths.get(folderPath, new String[0]);
        Files.createDirectories(path, new FileAttribute[0]);
        this.folderCreated = true;
    }

    private void spill(List<TsBlock> tsBlocks) throws IOException, IoTDBException {
        if (!this.folderCreated) {
            this.createFolder(this.folderPath);
        }
        String fileName = this.filePrefix + String.format("%05d", this.fileIndex) + FILE_SUFFIX;
        ++this.fileIndex;
        this.writeData(tsBlocks, fileName);
    }

    public void spillSortedData(List<SortKey> sortedData) throws IoTDBException {
        ArrayList<TsBlock> tsBlocks = new ArrayList<TsBlock>();
        TsBlockBuilder tsBlockBuilder = new TsBlockBuilder(this.dataTypeList);
        ColumnBuilder[] columnBuilders = tsBlockBuilder.getValueColumnBuilders();
        TimeColumnBuilder timeColumnBuilder = tsBlockBuilder.getTimeColumnBuilder();
        for (SortKey sortKey : sortedData) {
            this.writeSortKey(sortKey, columnBuilders, (ColumnBuilder)timeColumnBuilder);
            tsBlockBuilder.declarePosition();
            if (!tsBlockBuilder.isFull()) continue;
            tsBlocks.add(tsBlockBuilder.build());
            tsBlockBuilder.reset();
            timeColumnBuilder = tsBlockBuilder.getTimeColumnBuilder();
        }
        if (!tsBlockBuilder.isEmpty()) {
            tsBlocks.add(tsBlockBuilder.build());
        }
        try {
            this.spill(tsBlocks);
        }
        catch (IOException e) {
            throw new IoTDBException("Create file error: " + this.filePrefix + (this.fileIndex - 1) + FILE_SUFFIX, (Throwable)e, TSStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
        }
    }

    private void writeData(List<TsBlock> sortedData, String fileName) throws IOException, IoTDBException {
        Path filePath = Paths.get(fileName, new String[0]);
        Files.createFile(filePath, new FileAttribute[0]);
        try (FileChannel fileChannel = FileChannel.open(filePath, StandardOpenOption.WRITE);){
            for (TsBlock tsBlock : sortedData) {
                ByteBuffer tsBlockBuffer = this.serde.serialize(tsBlock);
                ByteBuffer length = ByteBuffer.allocate(4);
                length.putInt(tsBlockBuffer.capacity());
                length.flip();
                fileChannel.write(length);
                fileChannel.write(tsBlockBuffer);
            }
        }
        catch (IOException e) {
            throw new IoTDBException("Can't write intermediate sorted data to file: " + fileName, (Throwable)e, TSStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
        }
    }

    private void writeSortKey(SortKey sortKey, ColumnBuilder[] columnBuilders, ColumnBuilder timeColumnBuilder) {
        timeColumnBuilder.writeLong(sortKey.tsBlock.getTimeByIndex(sortKey.rowIndex));
        for (int i = 0; i < columnBuilders.length; ++i) {
            if (sortKey.tsBlock.getColumn(i).isNull(sortKey.rowIndex)) {
                columnBuilders[i].appendNull();
                continue;
            }
            columnBuilders[i].write(sortKey.tsBlock.getColumn(i), sortKey.rowIndex);
        }
    }

    public boolean hasSpilledData() {
        return this.fileIndex != 0;
    }

    private List<String> getFilePaths() {
        ArrayList<String> filePaths = new ArrayList<String>();
        for (int i = 0; i < this.fileIndex; ++i) {
            filePaths.add(this.filePrefix + String.format("%05d", i) + FILE_SUFFIX);
        }
        return filePaths;
    }

    public List<SortReader> getReaders(SortBufferManager sortBufferManager) throws IoTDBException {
        List<String> filePaths = this.getFilePaths();
        ArrayList<SortReader> sortReaders = new ArrayList<SortReader>();
        try {
            for (String filePath : filePaths) {
                sortReaders.add(new FileSpillerReader(filePath, sortBufferManager, this.serde));
            }
        }
        catch (IOException e) {
            throw new IoTDBException("Can't get file for FileSpillerReader, check if the file exists: " + filePaths, (Throwable)e, TSStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
        }
        return sortReaders;
    }

    public int getFileSize() {
        return this.fileIndex;
    }
}

